import * as path from 'path';
import { readFile } from 'node:fs/promises';
import { Context, SnapshotState, toMatchSnapshot } from 'jest-snapshot';
import { equals, iterableEquality, subsetEquality } from '@jest/expect-utils';
import * as matcherUtils from 'jest-matcher-utils';
import { expect } from 'expect';
import { pictoryProjectPath } from './utils';

const pictoryConfigPath = path.join(pictoryProjectPath, '_config.rsc');
export const updateSnapshots = process.env.SNAPSHOT_UPDATE !== undefined;

const wrapDashes = (msg = '', count = 80) => {
  const x = '-'.repeat((count - msg.length) / 2);
  return x + msg + x;
};

if (updateSnapshots) {
  console.warn(wrapDashes(' Warning! '));
  console.warn(
    'Updating snapshots - All tests will be successful. Review changes to snapshots',
  );
  console.warn(wrapDashes());
}

async function getPictoryConfig() {
  try {
    const fileContent = await readFile(pictoryConfigPath, 'utf-8');
    if (fileContent.length === 0) {
      return {};
    }
    return JSON.parse(fileContent);
  } catch (error) {
    console.error('Could not read pictory config:', error);
    throw new Error('Could not read pictory config');
  }
}

const pictoryTimestampRegex = /^\d{14}$/;
const pictoryGUIDRegex =
  /^[a-f\d]{8}-[a-f\d]{4}-[a-f\d]{4}-[a-f\d]{4}-[a-f\d]{12}$/;

/**
 * Build a Jest property matcher object that is used to override the default matchers for
 * certain properties.
 * This is used for dynamic values that will be different on each run
 * and would never match the value persistent in a previous snapshot
 * @see https://jestjs.io/docs/snapshot-testing#property-matchers
 *
 * Matchers for object nested in arrays must be defined for array element that is expected.
 * Therefore the "Devices" array is generated with the actual length
 */
function buildIgnorePropertiesMatcher(currentConfig: any) {
  if (Object.keys(currentConfig).length === 0) {
    return {};
  }

  const devicesLength = currentConfig.Devices?.length ?? 0;
  return {
    App: { saveTS: expect.stringMatching(pictoryTimestampRegex) },
    Devices: new Array(devicesLength).fill({
      GUID: expect.stringMatching(pictoryGUIDRegex),
    }),
  };
}

function getMatchSnapshotContext(t: TestController): Partial<Context> {
  const snapshotPath = `${t.fixture.path}.snapshot`;
  const snapshotState = new SnapshotState(snapshotPath, {
    updateSnapshot: updateSnapshots ? 'all' : 'new',
    snapshotFormat: {},
    rootDir: '',
    // set this to true to output full snapshots on error. by default only differences are shown
    expand: false,
  });

  return {
    snapshotState,
    currentTestName: `${t.fixture.name} - ${t.test.name}`,
    equals,
    utils: {
      ...matcherUtils,
      iterableEquality,
      subsetEquality,
    },
  };
}

export async function matchPictoryConfigSnapshot(t: TestController) {
  const currentValue = await getPictoryConfig();
  const matchSnapshotContext = getMatchSnapshotContext(t);
  const ignoreProperties = buildIgnorePropertiesMatcher(currentValue);

  const result = toMatchSnapshot.call(
    matchSnapshotContext,
    currentValue,
    ignoreProperties,
  );

  // Must always be called but `updateSnapshots` from above defines if snapshot files are actually written/replaced
  matchSnapshotContext.snapshotState.save();

  await t.expect(result.pass).ok(result.message());
}
