import { Selector } from 'testcafe';
import {
  AddDeviceBase,
  DeviceCatalog,
  DeviceIds,
  Editor,
} from './pictory.models';
import {
  getCatalogEntries,
  reloadCurrentPage,
  writeCustomCatalog,
} from './utils';
import { CatalogEntries } from './types';

/**
 * Initialize the test by:
 *  1. Writing the provided test data (if any) to custom-catalog.json
 *  3. Open pictory with the device catalog loaded
 *
 *  The test can then start asserting its tests conditions
 *
 * @param {TestController} t - The test controller object for the test.
 * @param {CatalogEntries} [customCatalogEntries] - Optional custom catalog entries to write.
 */
async function initTest(
  t: TestController,
  customCatalogEntries?: CatalogEntries,
): Promise<void> {
  if (customCatalogEntries !== undefined) {
    await writeCustomCatalog(customCatalogEntries);
  }
  await reloadCurrentPage(t);
}

/**
 * Asserts that all entries in the given catalogEntries are present in the tree view under the correct folder.
 *
 * @param {CatalogEntries} catalogEntries - The entries to check in the tree view.
 * @param {TestController} t - The TestController object for test assertions.
 */
async function assertAllEntriesAreInTreeView(
  catalogEntries: CatalogEntries,
  t: TestController,
) {
  for (const catalogEntry of catalogEntries) {
    const catalogFolder = DeviceCatalog.CatalogTree.find(
      `#ft_${catalogEntry.key}`,
    );
    await t.expect(catalogFolder.exists).ok();

    for (const childEntry of catalogEntry.children ?? []) {
      await t.expect(catalogFolder.find(`#ft_${childEntry.key}`).exists).ok();
    }
  }
}

let defaultCatalogEntries: CatalogEntries;

fixture(`Device Catalog`)
  // All tests in this fixture must set the catalog-custom.json BEFORE the page is loaded.
  // Therefore we must skip the login in the global "before" Test Hook and logging manually
  .beforeEach(async () => {
    // Read default catalog for each test
    defaultCatalogEntries = await getCatalogEntries();
  });

test('DeviceTree should show all devices from catalog.json', async (t) => {
  await initTest(t);

  await assertAllEntriesAreInTreeView(defaultCatalogEntries, t);
});

test('DeviceTree should show all devices from catalog-custom.json', async (t) => {
  const customEntries: CatalogEntries = [
    {
      key: 'test_folder_1',
      title: 'Test Folder 1',
      folder: true,
      children: [
        {
          key: 'Test_Device_1',
          title: 'Test Device 1 Title',
          tooltip: 'Test Device 1 Tooltip',
          icon: '##GSD_ICON_PATHNAME##',
        },
        {
          key: 'Test_Device_2',
          title: 'Test Device 2 Title',
          tooltip: 'Test Device 2 Tooltip',
          icon: '##GSD_ICON_PATHNAME##',
        },
      ],
    },
  ];
  await initTest(t, customEntries);

  // All default entries must be there
  await assertAllEntriesAreInTreeView(defaultCatalogEntries, t);
  // And also all custom entries must be there
  await assertAllEntriesAreInTreeView(customEntries, t);
});

test('DeviceTree should load without error if catalog-custom.json is an empty array', async (t) => {
  const customEntries: CatalogEntries = [];
  await initTest(t, customEntries);

  // All default entries must be rendered as usual
  await assertAllEntriesAreInTreeView(defaultCatalogEntries, t);
});

test('DeviceTree should NOT show devices from catalog-custom.json when the key also exists in default catalog.json', async (t) => {
  const firstDefaultCatalogDevice = defaultCatalogEntries[0].children[0];

  const customEntries: CatalogEntries = [
    {
      key: 'test_folder_1',
      title: 'Test Folder 1',
      folder: true,
      children: [
        {
          // Fake custom test device with the same key as th first device from the default catalog
          key: firstDefaultCatalogDevice.key,
          title: 'Test Device 1 Title',
          tooltip: 'Test Device 1 Tooltip',
          icon: '##GSD_ICON_PATHNAME##',
        },
        {
          key: 'Test_Device_2',
          title: 'Test Device 2 Title',
          tooltip: 'Test Device 2 Tooltip',
          icon: '##GSD_ICON_PATHNAME##',
        },
      ],
    },
  ];
  await initTest(t, customEntries);

  // All default entries must be rendered as usual
  await assertAllEntriesAreInTreeView(defaultCatalogEntries, t);

  const customCatalogTestFolder = DeviceCatalog.CatalogTree.find(
    `#ft_${customEntries[0].key}`,
  );
  const customCatalogDeviceWithDuplicateId = customEntries[0].children[0];
  //  First custom test device with duplicate key MUST NOT be found
  await t
    .expect(
      customCatalogTestFolder.find(
        `#ft_${customCatalogDeviceWithDuplicateId.key}`,
      ).exists,
    )
    .notOk();

  // Second custom test device with unique id should still be there
  const customCatalogDeviceWithUniqueId = customEntries[0].children[1];
  await t
    .expect(
      customCatalogTestFolder.find(`#ft_${customCatalogDeviceWithUniqueId.key}`)
        .exists,
    )
    .ok();
});

test('DeviceData table should show correct details of custom device', async (t) => {
  const customTestDevice = {
    // This device key is not used in the default entries but has a valid *.rap file
    key: 'OpenCV01_20160818_1_0',
    title: 'Test Device 1 Title',
    tooltip: 'Test Device 1 Tooltip',
    icon: '##GSD_ICON_PATHNAME##',
  };
  const customEntries: CatalogEntries = [
    {
      key: 'test_folder_1',
      title: 'Test Folder 1',
      folder: true,
      children: [customTestDevice],
    },
  ];
  await initTest(t, customEntries);

  await AddDeviceBase(DeviceIds.Connect4);
  // Add custom test device
  await t.dragToElement(
    Selector(`#ft_${customTestDevice.key}`),
    Editor.DropTargets.right,
  );

  await t
    .expect(Selector('#tblData_modulname_td_3').find('input').value)
    .eql(customTestDevice.title);
  await t
    .expect(Selector('#tblData_bmk_td_3').find('input').value)
    .eql(customTestDevice.title);
});
