const {
  backupPictoryProjectsDir,
  removePictoryProjectsBackupDir,
  restorePictoryProjectsDir, removeCustomCatalog, targetUrl, targetPassword,
} = require('./pictory/utils');
const { expectEmptyPictory, adminUser } = require('./common.models');
const { matchPictoryConfigSnapshot } = require("./pictory/utils.snapshots");


const chromiumDesktop = "chromium --ignore-certificate-errors --no-default-browser-check --unsafely-treat-insecure-origin-as-secure=http://localhost:59179,https://0.0.0.0:41443";
const chromiumHeadless = "chromium:headless --ignore-certificate-errors";
const showBrowser = process.env.SHOW_BROWSER !== undefined;
const showFirefox = process.env.FIREFOX !== undefined;
let browser = showBrowser ? chromiumDesktop : chromiumHeadless;
if (showFirefox) browser = "firefox";

module.exports = {
  hooks: {
    testRun: {
      // Global testRun hook `before` runs once before the actual test cases are executed
      before: async (ctx) => {
        ctx.meta = {
          targetUrl,
          targetPassword,
        }
        await backupPictoryProjectsDir();
      },

      // Global testRun hook `after` runs once after all test cases have been executed
      after: async () => {
        await restorePictoryProjectsDir();
        await removePictoryProjectsBackupDir();
        await removeCustomCatalog()
      },

    },

    test: {
      // Global test hook `before` runs before every single test case just like `test.beforeEach`
      before: async (t) => {
        // Before each and every test, we must login to webstatus and open pictory

        // 1. login and open pictory
        await t.useRole(adminUser);

        // 2. Assert empty pictory
        await expectEmptyPictory(t);
      },

      // Global test hook `after` runs after every single test case just like `test.afterEach`
      after: async (t) => {
        await matchPictoryConfigSnapshot(t);
        // After each test, restore the pictory projects directory and remove any custom-catalog.json to revert all changes made during the test
        await restorePictoryProjectsDir();
        await removeCustomCatalog()
      }
    },
  },
  compilerOptions: {
    typescript: {
      options: {
        esModuleInterop: true
      }
    }
    },
  src: ["tests/**/*.ts", "tests/studio-fixtures/**/*.js"],
  baseUrl: targetUrl,
  skipJsErrors: false,
  disablePageCaching: true,
  cache: false,
  testExecutionTimeout: 600000,
  // Abort testing after first error as the current state of the page is now undefined and follow-up errors may occur
  stopOnFirstFail: true,
  debugOnFail: false,
  // disable native automation because this does not support
  // opening new browser windows which is needed for node-red
  //disableNativeAutomation: false,
  // default - can be overwritten with CLI parameters on start
  browsers: [browser],
};
