import * as path from 'path';
import { access, readFile, unlink, writeFile } from 'node:fs/promises';
import { exec, execFile } from 'node:child_process';
import { promisify } from 'node:util';
import { CatalogEntries } from './types';
import { Context, SnapshotState, toMatchSnapshot } from 'jest-snapshot';
import { expect } from 'expect';

const execP = promisify(exec);

export const isDebug = process.env['DEBUG'] !== undefined;
export const isLocal = process.env['TEST_PICTORY_PATH'] !== undefined;
export const pictoryPath =
  process.env['TEST_PICTORY_PATH'] ?? '/var/www/revpi/pictory';
export const pictoryProjectPath = path.join(pictoryPath, 'projects');
export const pictoryProjectBackupPath = path.join(
  pictoryPath,
  'projects_backup',
);
export const targetUrl =
  process.env.TESTCAFE_TARGET_URL || 'https://0.0.0.0:41443';
export const targetPassword =
  process.env.TESTCAFE_TARGET_PASSWORD || 'raspberry';

/**
 * Backs up the original Pictory projects directory.
 * @returns {Promise<void>} A promise that resolves when the backup is complete.
 * @throws {Error} If the Pictory projects directory does not exist.
 */
export async function backupPictoryProjectsDir(): Promise<void> {
  try {
    await access(pictoryProjectPath);
  } catch (error) {
    throw new Error(
      `Could not find pictory projects dir at ${pictoryProjectPath}. ${error}`,
    );
  }

  await removePictoryProjectsBackupDir();
  await runCommand(`cp -a ${pictoryProjectPath} ${pictoryProjectBackupPath}`);
}

/**
 * Restores the Pictory Projects directory by removing the existing directory and copying the backup directory to its original location.
 *
 * @returns {Promise<void>} A Promise that resolves when the restoration is complete or rejects if an error occurs.
 */
export async function restorePictoryProjectsDir(): Promise<void> {
  await runCommand(`rm -rf ${pictoryProjectPath}`);
  await runCommand(`cp -a ${pictoryProjectBackupPath} ${pictoryProjectPath}`);
}

/**
 * Removes the backup directory for Pictory projects.
 * @returns Promise<void>
 */
export async function removePictoryProjectsBackupDir(): Promise<void> {
  await runCommand(`rm -rf ${pictoryProjectBackupPath}`);
}

export async function runCommand(command: string): Promise<void> {
  const { stderr } = await execP(command);
  if (stderr) {
    console.error(`Command '${command}' failed)`);
    console.error(stderr);
    throw new Error(`Command '${command}' failed)`);
  }
}

const dataPath = path.join(pictoryPath, 'resources/data');
const catalogPath = path.join(dataPath, 'catalog.json');
const catalogCustomPath = path.join(dataPath, 'catalog-custom.json');

export async function getCatalogEntries(): Promise<CatalogEntries> {
  const fileContent = await readFile(catalogPath, 'utf-8');
  return JSON.parse(fileContent) as CatalogEntries;
}

export async function writeCustomCatalog(
  entries: CatalogEntries,
): Promise<void> {
  await writeFile(catalogCustomPath, JSON.stringify(entries, null, 2), 'utf-8');
}

export async function removeCustomCatalog(): Promise<void> {
  return unlink(catalogCustomPath).catch(
    (e) =>
      isDebug &&
      console.debug('Error deleting catalog-custom.json', e.toString()),
  );
}

type ValidationResult = { isValid: boolean; error?: string };

export async function validatePictoryStartupConfig(): Promise<ValidationResult> {
  // If the tests are executed locally the revpimodio script will not work so we just return true
  if (isLocal) {
    return Promise.resolve({ isValid: true });
  }

  return new Promise<ValidationResult>((resolve) => {
    const scriptPath = path.join(
      process.cwd(),
      'tests/validate-start-config.py',
    );

    execFile(
      '/usr/bin/python3',
      [scriptPath],
      { encoding: 'buffer' },
      (error, stdout, stderr) => {
        if (error !== null) {
          return resolve({ isValid: false, error: error.message });
        }

        if (stderr?.length !== 0) {
          return resolve({ isValid: false, error: stderr.toString() });
        }

        resolve({ isValid: true });
      },
    );
  });
}

export async function reloadCurrentPage(t: TestController) {
  await t.setNativeDialogHandler((dialogType, message, url) => {
    if (dialogType === 'beforeunload') return;

    throw Error(
      `An unexpected ${dialogType} dialog with the message "${message}" appeared on ${url}.`,
    );
  });

  const currentUrl = await t.eval(() => document.documentURI);
  await t.navigateTo(currentUrl);
}

const pictoryConfigPath = path.join(pictoryProjectPath, '_config.rsc');

export async function matchPictoryConfigSnapshot(
  t: TestController,
  updateSnapshots: boolean,
) {
  const fileContent = await readFile(pictoryConfigPath, 'utf-8');
  const currentValue = JSON.parse(fileContent);
  const snapshotPath = `${t.fixture.path}.snapshot`;
  const snapshotState = new SnapshotState(snapshotPath, {
    updateSnapshot: updateSnapshots ? 'all' : 'new',
    snapshotFormat: {},
    rootDir: 'foo',
  });

  const matchSnapshotContext: Partial<Context> = {
    snapshotState,
    currentTestName: `${t.fixture.name} - ${t.test.name}`,
  };
  const result = toMatchSnapshot.call(matchSnapshotContext, currentValue, {
    saveTS: expect.any(Number),
  });

  // Must always be called but `updateSnapshots` from above defines if snapshot files are actually written/replaced
  snapshotState.save();

  await t.expect(result.pass).ok(result.message());
}
