import { Selector, t } from 'testcafe';
import { Dictionary } from './types';

type TMenu = {
  Menu: Selector;
  [index: string]: Selector;
};

export async function clickMenu<T extends TMenu>(
  t: TestController,
  x: T,
  key: keyof Omit<T, 'Menu'>,
): Promise<void> {
  return t.hover(x.Menu).click(x[key]);
}

export async function AddDevice(
  deviceId: string,
  location: EditorDeviceLocation,
): Promise<void> {
  const catalogSelector = getDeviceCatalogSelector(deviceId);
  const deviceSelector = getDeviceSelector(deviceId);
  const dropTargetSelector = Editor.DropTargets[location];
  const editorDeviceSelector = Editor.ConfiguredDevices[location];

  await t
    .dragToElement(catalogSelector, dropTargetSelector)
    .expect(deviceSelector.exists)
    .ok()
    .expect(editorDeviceSelector.id)
    .contains(deviceId);
}
export const AddDeviceLeft = (deviceId: string) => AddDevice(deviceId, 'left');
export const AddDeviceRight = (deviceId: string) =>
  AddDevice(deviceId, 'right');
export const AddDeviceBase = (deviceId: string) => AddDevice(deviceId, 'base');

// TODO: Generate from current catalog??
export const DeviceIds = {
  Core3: 'RevPiCore_20170404_1_2',
  Connect4: 'RevPiConnect4_20230409_1_0',
  Connect5: 'RevPiConnect5_20240315_1_0',
  DI: 'RevPiDI_20160818_1_0',
  DIO: 'RevPiDIO_20250109_1_1',
  MIO: 'RevPiMIO_20200901_1_0',
  RO: 'RevPiRO_20231018_1_0',
  Gateway_DMX: 'RevPiGateDMX_20160824_1_0',
  Gateway_EtherNetIP: 'RevPiGateEtherNetIP_20161012_1_0',
  Gateway_Profinet: 'RevPiGateProfinet_20160916_1_0',
};

export function getDeviceSelector(deviceId: string): Selector {
  return Selector(`#device_${deviceId}_001`);
}

export function getDeviceCatalogSelector(deviceId: string): Selector {
  return Selector(`#ft_${deviceId}`);
}

type EditorDeviceLocation = 'base' | 'left' | 'right';
const ConfiguredDevices: Dictionary<Selector, EditorDeviceLocation | 'all'> = {
  all: Selector('.deviceImage'),
  base: Selector('.deviceImage.isBase'),
  left: Selector('.deviceImage.isLeft'),
  right: Selector('.deviceImage.isRight'),
};

const DropTargets: Dictionary<Selector, EditorDeviceLocation> = {
  base: Selector('#dragTable td.dropCell'),
  left: Selector('#dragTable td.dropCell.edge').nth(0),
  right: Selector('#dragTable td.dropCell.edge').nth(1),
};

/**
 * Center editor area where devices can be dropped and configured devices are shown
 */
export const Editor = {
  DropTargets,
  ConfiguredDevices,
};

/**
 * Bottom Area with table for Inputs/Outputs config of currently selected device
 */
export const ValueEditor = {
  Table: Selector('#tblEdit'),
  getRow: (x: number) => ValueEditor.Table.find('tbody tr').nth(x),
  getColumnsOfRow: (x: number) => {
    const cols = ValueEditor.getRow(x).find('td');

    return {
      Type: cols.nth(0),
      Name: cols.nth(1),
      Value: cols.nth(2),
      Unit: cols.nth(3),
      Comment: cols.nth(4),
      Export: cols.nth(5),
    };
  },
};

/**
 * Left Area with the tree view of available devices
 */
export const DeviceCatalog = {
  CatalogTree: Selector('#catalogTree'),
  BaseFolder: getDeviceCatalogSelector('f_01'),
  GatewayFolder: getDeviceCatalogSelector('f_03'),
  IOFolder: getDeviceCatalogSelector('f_02'),
  VirtualDevicesFolder: getDeviceCatalogSelector('f_04'),
  VirtualModBusDevicesFolder: getDeviceCatalogSelector('f_05'),
};

export const FileMenu = {
  Menu: Selector('#file'),
  New: Selector('#file_new'),
  Clear: Selector('#file_clear'),
  Open: Selector('#file_open'),
  Save: Selector('#file_save'),
  SaveAs: Selector('#file_saveAs'),
  SaveAsStart: Selector('#file_saveAsStart'),
  Export: Selector('#file_export'),
  Exit: Selector('#file_exit'),
};

export const ToolsMenu = {
  Menu: Selector('#tools'),
  ManageConnections: Selector('#tools_manageConnections'),
  ResetServiceStates: Selector('#tools_resetServiceStates'),
  ResetLayout: Selector('#tools_resetLayout'),
  ResetDriver: Selector('#tools_resetDriver'),
  ScanDevices: Selector('#tools_scanDevices'),
  SetLanguage: Selector('#tools_setLanguage'),
  SetLanguage_DE: Selector('#tools_setLanguage_DE'),
  SetLanguage_EN: Selector('#tools_setLanguage_EN'),
  SetLanguage_ES: Selector('#tools_setLanguage_ES'),
  SetLanguage_FR: Selector('#tools_setLanguage_FR'),
  SetLanguage_IT: Selector('#tools_setLanguage_IT'),
  SetLanguage_PT: Selector('#tools_setLanguage_PT'),
  UserSettings: Selector('#tools_userSettings'),
};
