import { Selector } from 'testcafe';
import {
  AddDeviceBase,
  AddDeviceLeft,
  AddDeviceRight,
  clickMenu,
  DeviceIds,
  Editor,
  FileMenu,
  getDeviceCatalogSelector,
  getDeviceSelector,
  ValueEditor,
} from './pictory.models';
import { reloadCurrentPage, validatePictoryStartupConfig } from './utils';

fixture(`Pictory Device Handling`);

test('Add Core3 Base Device with MIO+DIO and save as start config', async (t) => {
  await AddDeviceBase(DeviceIds.Core3);
  await AddDeviceLeft(DeviceIds.DIO);
  await AddDeviceRight(DeviceIds.MIO);

  await clickMenu(t, FileMenu, 'SaveAsStart');
  await reloadCurrentPage(t);

  // Assert devices are still there after reloading the page
  await t
    .expect(Editor.ConfiguredDevices.base.exists)
    .ok()
    .expect(Editor.ConfiguredDevices.base.id)
    .contains(DeviceIds.Core3)

    .expect(getDeviceSelector(DeviceIds.DIO).exists)
    .ok()
    .expect(Editor.ConfiguredDevices.left.id)
    .contains(DeviceIds.DIO)

    .expect(getDeviceSelector(DeviceIds.MIO).exists)
    .ok()
    .expect(Editor.ConfiguredDevices.right.id)
    .contains(DeviceIds.MIO);

  const result = await validatePictoryStartupConfig();
  await t.expect(result.isValid).ok(result.error);
});

test('Add Connect4 with RO and save as named project and load it again', async (t) => {
  await AddDeviceBase(DeviceIds.Connect4);
  await AddDeviceRight(DeviceIds.RO);

  await clickMenu(t, FileMenu, 'SaveAs');

  await t
    .typeText('#txtInputFilename', 'connect4_ro')
    .click(Selector('#okDialogSaveProject span').withText('Ok'));

  await clickMenu(t, FileMenu, 'Clear');

  await t.expect(Editor.ConfiguredDevices.all.count).eql(0);

  await clickMenu(t, FileMenu, 'Open');

  await t
    .expect(Selector('#tblProjectFiles b').withText('connect4_ro').exists)
    .ok();
  await t.click(Selector('#dlgSaveProjectOpen_connect4_ro').withText('Open'));
});

test('Add Connect4 with DIO and set custom names/values for Input and Outputs', async (t) => {
  await AddDeviceBase(DeviceIds.Connect4);
  await AddDeviceLeft(DeviceIds.DIO);

  // Value Table should be visible
  await t.expect(ValueEditor.Table.exists).ok();

  // Row #1 should contain DIO's Input 1
  const firstInput = ValueEditor.getColumnsOfRow(0);
  await t.expect(firstInput.Type.innerText).eql('INP 1');
  //  Set custom name
  const input1Name = firstInput.Name.find('input');
  await t.expect(input1Name.value).eql('I_1');
  await t
    .selectText(input1Name)
    .pressKey('delete')
    .typeText(input1Name, 'CustomInput1')
    .pressKey('enter');

  // Row #33 should contain DIO's Output 1
  const firstOutput = ValueEditor.getColumnsOfRow(34);
  // First row's label should be type INPUT
  await t.expect(firstOutput.Type.innerText).eql('OUT 1');
  //  Set custom name
  const output1Name = firstOutput.Name.find('input');
  await t.expect(output1Name.value).eql('O_1');
  await t
    .selectText(output1Name)
    .pressKey('delete')
    .typeText(output1Name, 'CustomOutput1')
    .pressKey('enter');

  // Save and reload
  await clickMenu(t, FileMenu, 'SaveAsStart');
  await reloadCurrentPage(t);

  // Click DIO to open value editor
  await t.click(getDeviceSelector(DeviceIds.DIO));

  // Expect custom names still to be there
  await t.expect(input1Name.value).eql('CustomInput1');
  await t.expect(output1Name.value).eql('CustomOutput1');

  const result = await validatePictoryStartupConfig();
  await t.expect(result.isValid).ok(result.error);
});

test('Adding a Gateway left to Connect4 should show error message', async (t) => {
  await AddDeviceBase(DeviceIds.Connect4);

  await t.dragToElement(
    getDeviceCatalogSelector(DeviceIds.Gateway_DMX),
    Editor.DropTargets.left,
  );

  await t
    .expect(Selector('#lblDialogInfo').innerText)
    .contains('ERR_LEFT_INVALID_PRODUCTTYPE');
});
