#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""This script is used to set the PiCtory version number in all project files."""
import argparse
import sys
from collections import namedtuple
from glob import glob
from json import dumps, load
from os.path import dirname, join
from re import compile, sub

SemVer = namedtuple("SemVer", ["major", "minor", "patch"])


parser = argparse.ArgumentParser(
    description="Set new PiCtory version in all project files."
)
parser.add_argument("--force", action="store_true", help="Force version update.")
parser.add_argument("version", help="Version number x.y.z")
args = parser.parse_args()

re_semver = compile(r"^(?P<major>\d+)\.(?P<minor>\d+)\.(?P<patch>\d+)$")
working_dir = dirname(dirname(__file__))  # Root of PiCtory project

with open(join(working_dir, "src", "pictory", "config.json"), "r") as f:
    config_json = load(f)

old_version = re_semver.match(config_json["version"])
new_version = re_semver.match(args.version)


def error(msg):
    """
    Prints an error message to standard error output and exits the program.

    This function is used to output a specified error message to the standard error
    stream. Additionally, it terminates the execution of the program immediately with
    an exit code of 1.

    :param msg: The message to be displayed as an error. This parameter is of type ``str``.
    :return: This function does not return a value as it calls `exit(1)` to terminate
             the program after printing the error message.
    """
    print(f"Error: {msg}", file=sys.stderr)
    exit(1)


def check_higher_version() -> bool:
    """
    Determines if a new version is higher than an old version.

    This function compares two version strings represented by named groups ("major",
    "minor", and "patch"). It evaluates each component in order and determines
    if the new version is higher than the old version component by component.

    :rtype: bool
    :return: True if the new version is higher than or equal to the old version;
             otherwise, False.
    """
    old_semver = SemVer(*map(int, old_version.groups()))
    new_semver = SemVer(*map(int, new_version.groups()))

    # Check update stepping is max 1 version higher than the old version
    if new_semver.major > old_semver.major:
        return new_semver.major == old_semver.major + 1
    if new_semver.minor > old_semver.minor:
        return new_semver.minor == old_semver.minor + 1
    if new_semver.patch > old_semver.patch:
        return new_semver.patch == old_semver.patch + 1

    return False


if not new_version:
    error("Invalid version number. Must be in format x.y.z with digits only.")
    exit(1)

if not check_higher_version() and not args.force:
    error(
        f"New version is not valid!\n"
        f"    You can only update the version number if it is higher than the current\n"
        f"    version number {old_version.string}.\n\n"
        f"    In addition, the version may only be increased by one counter.\n\n"
        f"    Use --force to force update."
    )
    exit(1)

# Set version in config.json file
with open(join(working_dir, "src", "pictory", "config.json"), "w") as f:
    config_json["version"] = args.version
    f.write(dumps(config_json, indent=2))

# Set version in testcafe snapshots
snapshot_dir = join(working_dir, "test", "testcafe", "tests", "pictory")
lst_snapshot_files = glob(join(snapshot_dir, "*.snapshot"))
for snapshot_file in lst_snapshot_files:
    with open(join(snapshot_dir, snapshot_file), "r+") as f:
        snapshot = f.readlines()

        # Embedded config.rsc object must be formatted with new lines to match
        re_version_line = compile(r'\s*"version"\s*:\s*"\d+\.\d+\.\d+"\s*,\s*')
        for i in range(len(snapshot)):
            if re_version_line.match(snapshot[i]):
                snapshot[i] = sub(r"\d+\.\d+\.\d+", new_version.string, snapshot[i])

        f.seek(0)
        f.writelines(snapshot)
