# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2023-2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""Test MQTT system with a broker."""
from queue import Empty

from revpimodio2 import RevPiModIOSelected
from revpimodio2.modio import DevSelect

from _helpers import MqttResult, TestCaseMqtt


class MqttClientBroker(TestCaseMqtt):
    """Test client_mqtt.py with a broker."""

    pictory_file = "broker_tests.rsc"

    def test_sending_behaviour_cyclic(self):
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
        )
        self.assertEqual(3, len(rpi.device))

        # Use first device with shared ios = All existing
        self.create_client(rpi.device.export_all_cyclic_5_rw)

        # Change exported and not exported values, both must be received
        self.revpi_simulator.io.InputValue_1.value = 100
        self.revpi_simulator.io.I_1.value = True
        self.revpi_simulator.writeprocimg()

        lst_validated_messages = []
        while len(lst_validated_messages) != 2:
            # Wait about 5 seconds, this is the cycle frequency
            mqtt_result = self.message_queue.get(timeout=5.5)  # type: MqttResult

            # Shut down mqtt revpi client to do no more messages
            self.client.stop()

            if mqtt_result.topic == "dev/64/io/InputValue_1":
                self.assertEqual(100, mqtt_result.value)
                lst_validated_messages.append(mqtt_result)
            if mqtt_result.topic == "dev/64/io/I_1":
                self.assertEqual(1, mqtt_result.value)
                lst_validated_messages.append(mqtt_result)

        self.assertEqual(2, len(lst_validated_messages))

    def test_sending_behaviour_event(self):
        """This will test the setting 'Send on change'"""
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
        )
        self.assertEqual(3, len(rpi.device))

        # Use second device with shared ios = Marked as exported
        self.create_client(rpi.device.export_marked_event_ro)

        # Change exported and not exported values, only the exported should be received
        self.revpi_simulator.io.InputValue_1.value = 100
        self.revpi_simulator.io.I_1.value = True
        self.revpi_simulator.writeprocimg()

        lst_validated_messages = []
        while True:
            # Get event messages, they should be fired after changing
            try:
                # Use 10 seconds timeout to be sure no other messages arrives
                mqtt_result = self.message_queue.get(timeout=10.0)  # type: MqttResult
            except Empty:
                break
            lst_validated_messages.append(mqtt_result)

        self.assertEqual(1, len(lst_validated_messages))
        self.assertEqual(1, lst_validated_messages[0].value)

    def test_sending_behaviour_request(self):
        """This will test the setting 'Request with base_topic/get'"""
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
        )
        self.assertEqual(3, len(rpi.device))

        # Use second device with shared ios = Marked as exported
        self.create_client(rpi.device.export_marked_request_ro)

        with self.assertRaises(Empty):
            # Just wait, to check nothing arrives
            self.message_queue.get(timeout=10.0)

        # Request values of MQTT client device 66
        self.mqtt.publish("dev/66/get")

        any_message = self.message_queue.get(timeout=5.0)  # type: MqttResult
        self.assertTrue(any_message.topic.startswith("dev/66"))
