# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2023-2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""MQTT client for Revolution Pi."""

from logging import getLogger
from threading import Event
from time import perf_counter
from typing import List

from revpimodio2 import RevPiModIOSelected
from revpimodio2.errors import DeviceNotFoundError
from revpimodio2.modio import DevSelect

from mqtt_revpi_client.client_mqtt import MqttClient
from . import proginit as pi
from .watchdogs import ResetDriverWatchdog

log = getLogger(__name__)


class RevPiMqttClientManager:
    """Main program of RevPiMqttClientManager class."""

    def __init__(self):
        """Init RevPiMqttClientManager class."""
        log.debug("enter RevPiMqttClientManager.__init__")

        self._cycle_time = 1.0
        self.do_cycle = Event()
        self._running = True

        # Set class variables
        self.mqtt_clients = []  # type: List[MqttClient]
        self.mqtt_startup_timeout = 60.0

        log.debug("leave RevPiMqttClientManager.__init__")

    def mqtt_reload(self) -> int:
        """
        Stop all MQTT clients, create new ones and start them.

        All currently running MQTT clients will be shut down. The MQTT
        clients are created and started according to the PiCtory configuration
        with their settings. Afterward, all clients are waited for whether
        they have started or returned an error. The error_code can be used to
        identify the errors.

        The error code is bit-coded. The bits can be used to find out what
        caused an error.
        Error bits, if set, this error occurred.
            Bit 0 = Could not stop all MQTT clients
            Bit 1 = Cloud not load RevPiModIO configuration
            Bit 2 = Could not create one or more MQTT clients
            Bit 3 = One or more MQTT clients runs into startup timeout

        :returns: Error code (See description)
        """
        log.debug("enter RevPiMqttClientManager.mqtt_reload")

        # Shut down any existing mqtt client
        error_code = self.mqtt_stop()

        if not error_code:
            try:
                # Filter for 'MQTT Client' virtual devices, which have productType 24586
                modio = RevPiModIOSelected(
                    DevSelect(search_key="productType", search_values=("24586",)),
                    procimg=pi.pargs.procimg,
                    configrsc=pi.pargs.configrsc,
                    shared_procimg=True,
                )
            except DeviceNotFoundError:
                # No configured devices in PiCtory is not an error
                log.warning("No 'MQTT Client' devices configured in PiCtory.")

            except Exception as e:
                log.error(f"Can not load RevPiModIO: {e}")
                error_code |= 2

            else:
                # Start client instances
                for device in modio.device:
                    log.debug(f"Use virtual device on position {device.position}")
                    try:
                        client = MqttClient(device, revpimodio_debugging=pi.pargs.verbose >= 2)
                        client.start()
                    except Exception as e:
                        log.error(
                            f"Could not create MQTT client on position {device.position}: {e}"
                        )
                        error_code |= 4
                    else:
                        self.mqtt_clients.append(client)
                        log.info(f"Started mqtt client on virtual device {device.position}")

            # Set time until all instances must be started
            max_startup_time = perf_counter() + self.mqtt_startup_timeout
            for mqtt_client in self.mqtt_clients.copy():
                # Calculates the remaining waiting time that is available
                remaining_startup_timeout = max_startup_time - perf_counter()
                if remaining_startup_timeout < 0.0:
                    # This value must never be less than 0, as .wait would throw an exception
                    remaining_startup_timeout = 0.0

                # Wait until the client reported startup complete.
                if not mqtt_client.startup_complete.wait(remaining_startup_timeout):
                    log.error(
                        f"MQTT client start timeout exceeded for broker "
                        f"'{mqtt_client.broker_address}'"
                    )
                    mqtt_client.stop()
                    self.mqtt_clients.remove(mqtt_client)
                    error_code |= 8

        log.debug("leave RevPiMqttClientManager.mqtt_reload")
        return error_code

    def mqtt_stop(self) -> int:
        """Stop all MQTT clients."""
        log.debug("enter RevPiMqttClientManager.mqtt_stop")

        error_code = 0

        log.debug("Stop all mqtt clients")
        for mqtt_client in self.mqtt_clients:
            mqtt_client.stop()
            mqtt_client.join(5.0)
            if mqtt_client.is_alive():
                error_code = 1
                log.error(f"Could not stop mqtt client on broker '{mqtt_client.broker_address}'")

        # Remove all destroyed mqtt clients from list. This will be rebuilt on mqtt_start.
        self.mqtt_clients.clear()

        log.debug("leave RevPiMqttClientManager.mqtt_stop")
        return error_code

    @staticmethod
    def rotate_logfile() -> None:
        """Start a new logfile."""
        log.debug("enter RevPiMqttClientManager.rotate_logfile")

        pi.reconfigure_logger()
        log.warning("start new logfile")

        log.debug("leave RevPiMqttClientManager.rotate_logfile")

    def start(self) -> int:
        """Blocking mainloop of program."""
        log.debug("enter RevPiMqttClientManager.start")
        error_code = 0

        # Start all configured MQTT clients
        self.mqtt_reload()

        # Startup tasks
        driver_reset = ResetDriverWatchdog()

        pi.startup_complete()
        # Go into mainloop of daemon
        while self._running:
            ot = perf_counter()
            self.do_cycle.clear()

            # Main tasks of cycle loop
            if driver_reset.triggered:
                log.info("Driver reset triggered, reload MQTT clients")
                self.mqtt_reload()

            dm = divmod(ot - perf_counter(), self._cycle_time)
            # For float the result is (q, a % b), where q is usually math.floor(a / b) but may be 1 less than that.
            if dm[0] == -1.0:  # Cycle time not exceeded
                self.do_cycle.wait(dm[1])
            else:
                log.debug("Cycle time exceeded about {0:.0f} times".format(abs(dm[0])))

        # Cleanup tasks
        driver_reset.stop()
        self.mqtt_stop()

        log.debug("leave RevPiMqttClientManager.start")
        return error_code

    def stop(self) -> None:
        """Set stop request for mainloop."""
        log.debug("enter RevPiMqttClientManager.stop")

        self._running = False
        self.do_cycle.set()

        log.debug("leave RevPiMqttClientManager.stop")


def main() -> int:
    # Check all sub commands
    if pi.pargs.tools == "pictory":
        # Execution with sub-command pictory
        from custom_rap_installer import PiCtoryCommand, main as pictory_main
        from os.path import dirname, join

        source_dir = join(dirname(__file__), "rap_installer_source")
        return_code = pictory_main(
            (
                PiCtoryCommand(pi.pargs.pictory_commands)
                if pi.pargs.pictory_commands
                else PiCtoryCommand.INFO
            ),
            source_dir,
            pi.pargs.install_root,
            pi.pargs.remove if "remove" in pi.pargs else False,
            pi.pargs.kunbus_catalog,
        )

    else:
        # No subcommand used, this will start up the server (default)
        import signal

        root = RevPiMqttClientManager()

        # Set signals
        signal.signal(signal.SIGUSR1, lambda n, f: root.rotate_logfile())
        signal.signal(signal.SIGINT, lambda n, f: root.stop())
        signal.signal(signal.SIGTERM, lambda n, f: root.stop())

        return_code = root.start()

    pi.cleanup()
    return return_code
