# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""
MQTT client to send / receive IO values to a MQTT broker.

Inspired by the RevPiPyLoad project
https://github.com/naruxde/revpipyload/blob/b51c2b617a57cc7d96fd67e1da9f090a0624eacb/src/revpipyload/mqttserver.py
"""

from enum import IntEnum
from logging import getLogger
from os.path import join
from ssl import CERT_NONE
from threading import Event, Thread

import revpimodio2
from paho.mqtt.client import Client, connack_string
from revpimodio2.device import Device
from revpimodio2.device import Virtual

log = getLogger(__name__)


class SharedIos(IntEnum):
    """Translate enum values form pictory virtual device."""

    NONE = 0
    OWN = 1
    EXPORTED = 2
    ALL = 3


class MqttClient(Thread):
    def __init__(self, virtual_device: Virtual, revpimodio_debugging=False):
        """
        MQTT client with configuration in a virtual device.

        All published messages has a QoS level of 0 (fire and forget). The
        subscribed topics has QoS level 2. This does not cause the broker to
        buffer messages in the event of a disconnection, as the client
        connects to the broker without generating a session.

        :param virtual_device: Use this virtual device for configuration values
        :param revpimodio_debugging: Enable debugging of underlying RevPiModIO library
        """
        super().__init__()
        self.startup_complete = Event()

        # Remove the incrementing number of id to get the pure version string
        vdev_id = virtual_device.id.rsplit("_", maxsplit=1)[0]

        # This supports multiple versions of the mqtt virtual device
        if vdev_id == "device_MQTTRevPiClient_20231024_1_0":
            log.debug("PiCtory module: MQTTRevPiClient_20231024_1_0")

            # Set IO indexes, used while runtime in program
            self._io_index_client_status = 0
            self._io_index_disconnect_client = 1

            # Read all MEM configurations
            self._broker_address = virtual_device[2].defaultvalue
            self._port = virtual_device[3].defaultvalue
            basetopic = virtual_device[4].defaultvalue
            self._shared_ios = SharedIos(virtual_device[5].defaultvalue)
            sending_behavior = virtual_device[6].defaultvalue
            self._write_outputs = bool(virtual_device[7].defaultvalue)
            self._username = virtual_device[8].defaultvalue
            self._password = virtual_device[9].defaultvalue
            self._use_tls_encryption = bool(virtual_device[10].defaultvalue)
            self._tls_certificate_path = virtual_device[11].defaultvalue  # todo: Implement
            self._tls_key_path = virtual_device[12].defaultvalue  # todo: Implement
            self._replace_io = virtual_device[13].defaultvalue

        else:
            raise RuntimeError(
                f"The device version '{virtual_device.id}' of mqtt client is not supported"
            )

        self._evt_data = Event()
        self._evt_exit = Event()
        self._exported_ios = []
        self._send_events = sending_behavior == 255
        self._sendinterval = 0 if self._send_events else sending_behavior

        # Module driver to set client status and read disconnect flag
        modio_module = revpimodio2.RevPiModIODriver(
            virtual_device.position,
            procimg=virtual_device._modio.procimg,
            configrsc=virtual_device._modio.configrsc,
        )
        self._me_module = modio_module.device[virtual_device.position]  # type: Virtual
        self._me_module.setdefaultvalues()
        self._me_module.writeprocimg()
        self._me_module._modio.debug = 1 if revpimodio_debugging else -1

        self._modio = revpimodio2.RevPiModIO(
            procimg=virtual_device._modio.procimg,
            configrsc=virtual_device._modio.configrsc,
            autorefresh=self._send_events,
            monitoring=not self._write_outputs,
            replace_io_file=self._replace_io,
            shared_procimg=True,
        )
        self._modio.debug = 1 if revpimodio_debugging else -1
        self._set_client_status_value(255)

        # Filter IOs to share, defined with "shared_io" MEM setting
        if self._shared_ios is not SharedIos.NONE:
            for dev in self._modio.device:  # type: Device
                if self._shared_ios is SharedIos.OWN and dev.position != virtual_device.position:
                    # Do not share IOs from devices other than your own
                    continue

                for io in dev.get_allios(
                    export=True if self._shared_ios is SharedIos.EXPORTED else None
                ):
                    # Add IO to our working list and register change event
                    io.reg_event(self._evt_io)
                    self._exported_ios.append(io)

        # Export special IOs of core device like a1green and so on
        lst_coreio = []
        if self._shared_ios is not SharedIos.NONE and self._shared_ios is not SharedIos.OWN:
            if self._modio.core:
                for obj_name in dir(self._modio.core):
                    # Scan all non-private objects of the core device
                    if obj_name.find("_") == 0:
                        continue
                    obj = getattr(self._modio.core, obj_name)
                    if isinstance(obj, revpimodio2.io.IOBase) and obj.export:
                        lst_coreio.append(obj)

            for io in lst_coreio:
                # Add IO to our working list and register change event
                io.reg_event(self._evt_io)
                self._exported_ios.append(io)

        # Prepare all used topics
        self._basetopic = basetopic.strip("/")
        self._mqtt_io = join(self._basetopic, "io/{0}")
        self._mqtt_ioset = join(self._basetopic, "set/#")
        self._mqtt_ioreset = join(self._basetopic, "reset/#")
        self._mqtt_senddata = join(self._basetopic, "get")

        self._mq = Client()

    def _evt_io(self, name: str, value) -> None:
        """Event callback, if sending behavior is set to 'Send on change'."""
        if not self._mq.is_connected():
            # As long as we only support QoS=0, we do not have to do a mqtt publish
            return

        if isinstance(value, bytes):
            # Send bytes as string list with the unsigned integer values of each byte
            value = " ".join([str(single_byte_int) for single_byte_int in value])
        if isinstance(value, bool):
            # Convert False/True to 0/1. Publish function would send the string.
            value = int(value)

        self._mq.publish(self._mqtt_io.format(name), value, 0)

    def _set_client_status_value(self, error_code: int) -> None:
        """Update 'client status' input."""
        self._me_module[self._io_index_client_status].value = error_code
        self._me_module.writeprocimg()
        log.debug(f"Set client status output to {error_code}")

    def _on_connect(self, client, userdata, flags, rc) -> None:
        """
        The callback called when the broker responds to our connection request.

        :param client: The client instance for this callback
        :param userdata: The private user data as set in Client() or userdata_set()
        :param flags: Is a dict that contains response flags from the broker
        :param rc: Connect status code
            0: Connection successful
            1: Connection refused - incorrect protocol version
            2: Connection refused - invalid client identifier
            3: Connection refused - server unavailable
            4: Connection refused - bad username or password
            5: Connection refused - not authorised
        :see: https://eclipse.dev/paho/files/paho.mqtt.python/html/client.html#paho.mqtt.client.Client.on_connect
        """
        log.debug("enter MqttClient._on_connect()")

        if rc > 0:
            log.warning(
                f"Can not connect to mqtt broker '{self._broker_address}' - "
                f"error '{connack_string(rc)}' - will retry"
            )
        else:
            # Subscribe IOs with fix QoS 2
            client.subscribe(self._mqtt_senddata, 2)
            if self._write_outputs:
                client.subscribe(self._mqtt_ioset, 2)
                client.subscribe(self._mqtt_ioreset, 2)
            log.debug("Subscribed topics to trigger internal events")

        self._set_client_status_value(rc)

        log.debug("leave MqttClient._on_connect()")

    def _on_disconnect(self, client, userdata, rc):
        """MQTT callback for a disconnect from broker."""
        log.debug("enter MqttClient._on_disconnect()")

        self._set_client_status_value(255)

        if rc != 0:
            log.warning("Unexpected disconnection from mqtt broker - will try to reconnect")

        log.debug("leave MqttClient._on_disconnect()")

    def _on_message(self, client, userdata, msg) -> None:
        """MQTT callback for a received message."""
        log.debug("Received topic: {0}".format(msg.topic))

        # On a pure "get" request topic, we just send all shared IOs
        if msg.topic == self._mqtt_senddata:
            # Set event to let the mainloop send all IO data
            self._evt_data.set()
            return

        # Process actions, based on IO names like set, reset,

        # The I/O name may contain forward slashes. Those look nice on the
        # MQTT bus, but make parsing the topic for actions a bit harder since
        # we cannot simply split the topic and know at what index to find the
        # action. To find the action we remove base topic parts to get the
        # action and finally reassemble the I/O name with slashes.
        action_topic = msg.topic.removeprefix(self._basetopic)
        action_topic = action_topic.strip("/")
        lst_topic = action_topic.split("/")
        if len(lst_topic) < 2 or lst_topic[0].lower() not in ("get", "set", "reset"):
            log.error(
                f"Wrong format for topic '{msg.topic}', "
                f"expected '{self._basetopic}/[get|set|reset]/<ioname>'"
            )
            return

        io_action = lst_topic[0]
        ioset = io_action == "set"
        ioreset = io_action == "reset"
        ioname = "/".join(lst_topic[1:])
        coreio = ioname.startswith("core.")

        try:
            if coreio:
                coreio = ioname.split(".")[-1]
                io = getattr(self._modio.core, coreio)
                if not isinstance(io, revpimodio2.io.IOBase):
                    raise RuntimeError()
            else:
                io = self._modio.io[ioname]

            io_needbytes = type(io.value) == bytes

        except Exception:
            log.error(f"Can not find io '{ioname}'")
            return

        if io not in self._exported_ios:
            log.error(f"IO '{ioname}' is not included in 'Shared IOs' setting of MQTT module")

        elif ioset and io.type != revpimodio2.OUT:
            log.error(f"Can not write to input '{ioname}'")

        elif ioset:
            # Convert MQTT payload to valid output value
            value = msg.payload.decode("utf8")

            if value.isdecimal():
                value = int(value)

                if io_needbytes:
                    try:
                        # Try to convert int to bytes, if IO is a byte type
                        value = value.to_bytes(io.length, io.byteorder)
                    except OverflowError:
                        log.error(
                            "Can not convert value '{value}' to fitting bytes for output '{ioname}'"
                        )
                        return

            elif value.lower() == "false" and not io_needbytes:
                value = 0
            elif value.lower() == "true" and not io_needbytes:
                value = 1
            else:
                log.error(f"Can not convert value '{value}' for output '{ioname}'")
                return

            # Write Value to RevPi process image
            try:
                io.value = value
                # Write data, if we do not use the event handler of RevPiModIO
                if not self._send_events:
                    io._parentdevice.writeprocimg()
                log.debug(f"Set output '{io.name}' to value '{io.value}'")
            except Exception:
                log.error(f"Could not write value '{value}' to output '{ioname}'")

        elif ioreset:
            # Counter zurücksetzen
            if not isinstance(io, revpimodio2.io.IntIOCounter):
                log.warning(f"IO '{ioname}' is not a counter")
            else:
                io.reset()

    def mqtt_connect(self) -> None:
        """Connect to MQTT broker."""

        # Always create a new instance, because you can not reuse a disconnected one
        if self._mq.is_connected():
            self._mq.disconnect()
        self._mq = Client()

        if self._username and self._password:
            self._mq.username_pw_set(self._username, self._password)
        if self._use_tls_encryption:
            self._mq.tls_set(cert_reqs=CERT_NONE)
            self._mq.tls_insecure_set(True)

        # Set callback functions of this class to MQTT client
        self._mq.on_connect = self._on_connect
        self._mq.on_disconnect = self._on_disconnect
        self._mq.on_message = self._on_message

        # First try to connect synchronous to MQTT broker
        log.info("Connecting to mqtt broker {0}".format(self._broker_address))
        try:
            self._mq.connect(self._broker_address, self._port, keepalive=60)
        except Exception:
            # Fallback to async connection, if broker is not yet available
            self._on_connect(self._mq, None, None, 3)
            self._mq.connect_async(self._broker_address, self._port, keepalive=60)

        # Start internal loop for reconnect and receive topics
        self._mq.loop_start()

    def run(self):
        """Mainloop of MQTT Client."""
        log.debug("enter MqttClient.run()")

        # This will start the event monitoring of RevPiPyLoad
        if self._send_events:
            log.debug("Start non blocking mainloop of revpimodio")
            self._modio.mainloop(blocking=False)

        self.startup_complete.set()

        cycle_counter = 0
        mrk_disconnect_output = True
        while not self._evt_exit.is_set():

            # Check the disconnect output of this mqtt module
            self._me_module.readprocimg()
            if self._me_module[1].value and not mrk_disconnect_output:
                mrk_disconnect_output = True
                log.debug(
                    f"'Disconnect client' output set, will disconnect from "
                    f"broker {self._broker_address} now"
                )
                self._mq.disconnect()
            elif not self._me_module[1].value and mrk_disconnect_output:
                mrk_disconnect_output = False
                log.debug(
                    f"'Disconnect client' output not set, will connect to "
                    f"broker {self._broker_address} now"
                )
                self.mqtt_connect()

            # Send IOs due to time or request event
            if cycle_counter and cycle_counter == self._sendinterval or self._evt_data.is_set():
                self._evt_data.clear()

                # Read IO values, if we do not use the event handler of RevPiModIO
                if not self._send_events:
                    self._modio.readprocimg()

                # Send all selected IOs (PiCtory device setting 'Shared IOs')
                for io in self._exported_ios:
                    # Use _evt_io function to set the same format for MQTT
                    self._evt_io(io.name, io.value)

            # Value of _sendinterval is 0, if 'Sending behavior' is 'Send on change'
            cycle_counter += 1
            if cycle_counter > self._sendinterval:
                cycle_counter = 0

            self._evt_exit.wait(1.0)

        self._mq.disconnect()
        self._set_client_status_value(255)
        log.info("Disconnected from mqtt broker {0}".format(self._broker_address))

        self._modio.cleanup()

        log.debug("leave MqttClient.run()")

    def stop(self):
        """
        Request to stop the MQTT client.

        You have to check .is_alive or .join this object to check the end of
        execution.
        """
        log.debug("enter MqttClient.stop()")
        self._evt_exit.set()
        log.debug("leave MqttClient.stop()")

    @property
    def broker_address(self) -> str:
        """Get the broker address of this client."""
        return self._broker_address

    @property
    def connected(self) -> bool:
        """Get connection state of mqtt client."""
        return self._me_module[self._io_index_client_status].value == 0
