// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";
import { fileExists, readFromFile, writeToFile } from "../common/helper.js";
import { isServiceActive, reloadSystemd, restartService } from "../common/systemd-tools.js";
import { NODE_RED_SERVICE } from "./cards.jsx";

// checksum of nodered's default settings.js
export const defaultSettingsJsHash = "49ef576d2920ea75b84ea39e078b7d9f";

// file paths
// eslint-disable-next-line no-undef
const PLUGIN_DATA_DIR = `${__APP_DATA_DIR__}/revpi-nodered/`;
const LOCAL_DB_PATH = `${PLUGIN_DATA_DIR}/localDb.json`;

// Default path to nodered settings.js
let SETTINGS_JS_PATH = "/var/lib/revpi-nodered/.node-red/settings.js";
export const TEMPLATE_SETTINGS_PATH = "settings/template-settings";

/**
 * Prepare all paths and load data
 */
export async function init () {
    // Creates the app data dir
    await cockpit.spawn(["mkdir", "-p", PLUGIN_DATA_DIR], { superuser: "require" });
    // Load node settings.js relative to home directory
    try {
        const nodeRedHome = await cockpit.script("echo -n ~nodered");
        SETTINGS_JS_PATH = `${nodeRedHome}/.node-red/settings.js`;
    } catch (err) {
        console.warn("Failed to resolve Node-RED settings directory:", err);
    }
    await loadDb();
}

/**
 * Enum-like object representing the two types of settings configurations.
 * Used to avoid hardcoded string literals when calling functions like `writeSettings`.
 */
export const SettingTypes = {
    SIMPLE: "simple",
    ADVANCED: "advanced"
};

// In-memory localDb to save UI state
let localDb = {};

const localDbDefaults = {
    useAuthentication: true, // defaults to true
    users: [], // array of user objects, { username: "admin", password: "pwHash", permissions: "read or *" }
    simpleHash: "", // checksum to detect whether settings.js file has changed in simple-mode
    advancedHash: "", // checksum to detect whether settings.js file has changed in advanced-mode
    advancedSettings: null, // Content of code editor in advanced tab
    savedSettingsMode: null // null or SettingTypes.SIMPLE or SettingTypes.ADVANCED
};

/**
 * Check if a user with the given username exists in the local DB.
 * @param {string} userName
 * @returns {boolean}
 */
export function userExists (userName) {
    return localDb.users.some((localUser) => localUser.username === userName);
}

/**
 * Returns true if either hash is set, indicating settings were applied at least once.
 * @returns {boolean}
 */
export function anyHashExists () {
    return !(localDb.simpleHash === "" && localDb.advancedHash === "");
}

/**
 * Adds a new user with hashed password and permissions to the local DB.
 * @param {string} username
 * @param {string} password - plain text password
 * @param {"read"|"*"} permissions
 */
export async function addUser (username, password, permissions) {
    if (!["*", "read"].includes(permissions)) throw new Error("Permissions must be one of read or *");
    if (userExists(username)) throw new Error("Username already exists in localDb");
    const hashPassword = await generateNodeRedPassword(password);
    const user = {
        username,
        password: hashPassword,
        permissions
    };
    localDb.users.push(user);
}

/**
 * Removes a user from the local DB by username.
 * @param {string} userName
 */
export function removeUser (userName) {
    if (userExists(userName)) localDb.users = localDb.users.filter((user) => user.username !== userName);
}

/**
 * Updates an existing user's password or permissions in the local DB.
 * @param {string} userName
 * @param {object} update - Can include `password` and/or `permissions`
 */
export async function updateUser (userName, {
    password,
    permissions
}) {
    const userIndex = localDb.users.findIndex((user) => user.username === userName);

    if (userIndex === -1) {
        throw new Error("User not found. Cannot update");
    }

    const updatedUser = { ...localDb.users[userIndex] };

    if (permissions) {
        if (!["*", "read"].includes(permissions)) {
            throw new Error("Permissions must be one of read or *");
        }
        updatedUser.permissions = permissions;
    }

    if (password) {
        updatedUser.password = await generateNodeRedPassword(password);
    }

    localDb.users[userIndex] = updatedUser;
}

/**
 * Sets a top-level key in the local DB to a new value.
 * Throws if key is `users`.
 * @param {string} key
 * @param {*} newValue
 */
export function setLocalDbValue (key, newValue) {
    if (!(key in localDb)) throw new Error("Property does not exist on localDb");
    if (key === "users") throw new Error("Users must be added via addUser");
    localDb[key] = newValue;
}

/**
 * Gets a value from the local DB by key.
 * @param {string} key
 * @returns {*}
 */
export function getLocalDbValue (key) {
    if (!(key in localDb)) return new Error("Property does not exist on localDb");
    return localDb[key];
}

/**
 * Compares in-memory localDb to saved file on disk.
 * @returns {boolean} true if identical
 */
export async function isLocalDbInSync () {
    const memoryState = JSON.stringify(localDb);
    // cockpit file read
    const content = await readFromFile(LOCAL_DB_PATH);

    const settingsJsFileExists = await fileExists(SETTINGS_JS_PATH);

    return settingsJsFileExists && memoryState === content;
}

/**
 * Writes a formatted Node-RED settings.js file with user data.
 * Stores the generated hash into localDb.
 * @param {string} settingsType "simple" or "advanced"
 */
export async function writeSettings (settingsType) {
    try {
        await cockpit.script(
            `
            # Create settings.js directory if it does not exist
            dir_path="$(dirname '${SETTINGS_JS_PATH}')"
            if [ ! -d "$dir_path" ]; then
                mkdir -p "$dir_path"
                chown nodered: "$dir_path"
            fi
            `,
            { superuser: "require" }
        );
        if (settingsType === SettingTypes.SIMPLE) {
            const permissions = localDb.useAuthentication ? "" : "permissions: \"*\"";
            const users = JSON.stringify(localDb.users);
            let simpleSettingsData = await readFromFile("settings/simple-settings");
            simpleSettingsData = cockpit.format(simpleSettingsData, {
                adminAuth_default_permissions: permissions,
                adminAuth_users: users
            });

            await writeToFile(SETTINGS_JS_PATH, simpleSettingsData);

            // Store simple hash to localDb
            localDb.simpleHash = await getSettingsJsChecksum();
            localDb.savedSettingsMode = SettingTypes.SIMPLE;
            await saveDb();
        } else if (settingsType === SettingTypes.ADVANCED) {
            await writeToFile(SETTINGS_JS_PATH, localDb.advancedSettings);

            // Store advanced hash to localDb
            localDb.advancedHash = await getSettingsJsChecksum();
            localDb.savedSettingsMode = SettingTypes.ADVANCED;
            await saveDb();
        } else {
            console.error("Invalid settingsType:", settingsType);
            throw new Error("Invalid settingsType");
        }
    } catch (error) {
        console.error("Error in writeSettings:", error);
        throw error; // Re-throw error to propagate it up
    }
}

/**
 * Saves the current localDb to disk as JSON.
 */
export async function saveDb () {
    const dbString = JSON.stringify(localDb);
    await writeToFile(LOCAL_DB_PATH, dbString);
}

/**
 * Loads the localDb from disk and sets it in memory.
 */
export async function loadDb () {
    // cockpit file read
    const content = await readFromFile(LOCAL_DB_PATH);
    if (content !== null) {
        try {
            localDb = JSON.parse(content);
        } catch (error) {
            console.error(error);
        }
    }

    // Set default db values if not loaded from localDb file
    for (const prop in localDbDefaults) {
        if (!(prop in localDb)) {
            localDb[prop] = localDbDefaults[prop];
        }
    }
    // Init default value for advancedSettings. We have to check this to merge new values
    // to our database.
    if (localDb.advancedSettings === null) {
        localDb.advancedSettings = await readFromFile(TEMPLATE_SETTINGS_PATH);
    }
}

/**
 * Gets an MD5 checksum of the current settings.js file.
 * If settings.js does not exist, returns undefined.
 * @returns {string | undefined} The checksum if available, otherwise undefined.
 */
export async function getSettingsJsChecksum () {
    try {
        const md5sum = await cockpit.spawn(["md5sum", SETTINGS_JS_PATH], { superuser: "try" });
        return md5sum.split(" ")[0];
    } catch (error) {
        console.error(error);
    }
}

/**
 * Load the content of the current Node Red settings.js
 * @returns {string|null} File content if exists, otherwise null.
 */

export async function readSettingsJsContent () {
    try {
        return readFromFile(SETTINGS_JS_PATH);
    } catch (error) {
        console.error(error);
        return null;
    }
}

/**
 * Generates a hashed password string suitable for Node-RED authentication.
 *
 * @param {string} plainPassword - The plain text password to be hashed.
 * @return {Promise<string>} A promise that resolves to the hashed password as a string.
 * @throws {Error} If an error occurs during password hashing.
 */
async function generateNodeRedPassword (plainPassword) {
    try {
        const pwHash = await cockpit.script(`echo -n ${plainPassword} | /usr/share/revpi-nodered/node_modules/node-red/bin/node-red-pi admin hash-pw`);
        return pwHash.trim().split(" ").at(-1);
    } catch (error) {
        console.error(error);
        throw new Error("Error in generateNodeRedPassword");
    }
}

/**
 * Checks if the "nodered" user is a member of the groups "audio", "video", and "render".
 * This is used to determine if Node-RED can use audio and video resources.
 *
 * @return {Promise<boolean>} A promise that resolves to true if the "nodered" user is a member of all three groups.
 */
export async function canNoderedUseAudioVideo () {
    try {
        const output = await cockpit.script("id -Gn nodered");
        // output is a string with groups, e.g. "user adm dialout cdrom"
        const memberGroups = output.split(/\s+/);
        return ["audio", "video", "render"].every((g) => memberGroups.includes(g));
    } catch (error) {
        console.error("Failed to check audio/video groups:", error);
        return false;
    }
}

/**
 * Checks if the user nodered has access to serial interfaces by verifying
 * membership in the "dialout" group.
 *
 * @return {Promise<boolean>} A promise that resolves to true if the user
 * nodered is in the "dialout" group, allowing access to serial interfaces.
 * Resolves to false if the user is not in the group or if an error occurs.
 */
export async function canNoderedUseSerialInterfaces () {
    try {
        const output = await cockpit.script("id -Gn nodered");
        // output is a string with groups, e.g. "user adm dialout cdrom"
        return output.split(/\s+/).includes("dialout");
    } catch (error) {
        console.error("Failed to check dialout group:", error);
        return false;
    }
}

/**
 * Removes the user "nodered" from the given groups using system commands.
 *
 * This function executes a shell command to remove the "nodered" user from
 * the system given groups. Administrative permissions are required, and
 * the operation is executed with elevated privileges.
 *
 * @param {string[]} groupsToRemove - An array of group names to remove the user from.
 * @return {Promise<boolean>} Resolves to true if the operation is successful,
 * or false if an error occurs during the process.
 */
export async function removeNoderedFromGroups (groupsToRemove) {
    try {
        for (const group of groupsToRemove) {
            await cockpit.spawn(["gpasswd", "-d", "nodered", group], { superuser: "try" });
        }
        await restartNodeRedService();
        return true;
    } catch (error) {
        console.error("Failed to remove nodered from dialout group:", error);
        return false;
    }
}

/**
 * Adds the user "nodered" to the given groups by executing a system command with superuser privileges.
 *
 * @param {string[]} groupsToAdd - An array of group names to add the user to.
 * @return {boolean} Returns true if the user was successfully added to the group. Logs an error and returns undefined if the operation fails.
 */
export async function addNoderedToGroups (groupsToAdd) {
    try {
        for (const group of groupsToAdd) {
            await cockpit.spawn(["gpasswd", "-a", "nodered", group], { superuser: "try" });
        }
        await restartNodeRedService();
        return true;
    } catch (error) {
        console.error("Failed to add nodered to dialout group:", error);
    }
}

/* Path variables used in activateSandbox and deactivateSandbox */
const noderedServiceDDir = "/etc/systemd/system/nodered.service.d";
const sandboxOverrideConfigFilePath = `${noderedServiceDDir}/disable-sandbox-override.conf`;
/**
 * Checks if the current environment is a sandbox.
 *
 * This method determines whether the sandbox mode is active by
 * checking the existence of a specific override configuration file.
 *
 * @return {Promise<boolean>} A promise that resolves to `true` if the environment is in sandbox mode, otherwise `false`.
 */
export async function isSandbox () {
    return !await fileExists(sandboxOverrideConfigFilePath);
}
/**
 * Activates the sandbox by removing the override configuration file, reloading the systemd manager configuration,
 * and restarting the necessary service.
 *
 * @return {Promise<void>} A promise that resolves when the sandbox has been successfully activated or rejects with an error if the activation fails.
 */
export async function activateSandbox () {
    try {
        await cockpit.spawn(["rm", "-f", sandboxOverrideConfigFilePath], { superuser: "try", err: "message" });
        await reloadSystemd();
        await restartNodeRedService();
    } catch (error) {
        console.error("Failed to activate sandbox:", error);
        throw error;
    }
}

/**
 * Deactivates the systemd sandbox for a specified service by creating
 * and writing a configuration file that resets systemd sandboxing
 * settings to their defaults (disabled). It then reloads systemd
 * configurations and restarts the service.
 *
 * @return {Promise<void>} A promise that resolves when the sandbox deactivation
 * and service restart are successfully completed. Rejects with an error if */
export async function deactivateSandbox () {
    try {
        await cockpit.spawn(["mkdir", "-p", noderedServiceDDir], { superuser: "try", err: "message" });
        // write this into the file disable-sandbox-override.conf
        const sandboxConfig =
`[Service]
# Reset systemd sandboxing to defaults (disabled)
ProtectSystem=no
ProtectControlGroups=no
ProtectKernelModules=no
ProtectKernelTunables=no
# Restrict file system access to the following directories
InaccessiblePaths=
`;
        await writeToFile(sandboxOverrideConfigFilePath, sandboxConfig);
        await reloadSystemd();
        await restartNodeRedService();
    } catch (error) {
        console.error("Failed to deactivate sandbox:", error);
        throw error;
    }
}

/**
 * Restarts the Node-RED service if it is currently active.
 * This function checks the status of the Node-RED service, and if it is active, it proceeds to restart it.
 *
 * @return {Promise<void>} A promise that resolves when the service has successfully restarted or if the service was not active.
 */
export async function restartNodeRedService () {
    const serviceActive = await isServiceActive(NODE_RED_SERVICE);
    if (!serviceActive) return;
    await restartService(NODE_RED_SERVICE);
}
