// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import {
    Button,
    Flex,
    FlexItem, GridItem, Text, TextContent,
    Tooltip
} from "@patternfly/react-core";
import React from "react";
import { setLocalDbValue } from "./revpi-nodered.settings.js";
// Code Editor imports
import { Controlled as CodeMirror } from "react-codemirror2";
// Make sure to import codemirror CSS and theme
import "codemirror/lib/codemirror.css";
// Import language mode (e.g., XML/HTML)
import "codemirror/mode/javascript/javascript";

import cockpit from "cockpit";
import { getServicePid, subscribeToUnitChanges } from "../common/systemd-tools.js";
import { getBindPorts } from "../common/helper.js";
import { NODE_RED_SERVICE, NODES_SERVER_SERVICE } from "./cards.jsx";
import { CompressIcon, ExpandIcon } from "@patternfly/react-icons";
import { SwitchWithSpinner } from "../common/ui-components.jsx";

const _ = cockpit.gettext;

const MIN_EDITOR_HEIGHT = 450;
const VIEWPORT_OFFSET = 572;
const FULLSCREEN_VIEWPORT_OFFSET = 0;

// CodeEditor component: A wrapper around CodeMirror for editing JavaScript code
export const CodeEditor = ({ code, setCode, updateSettingsChanges, isFullscreen }) => {
    const editorRef = React.useRef(null);
    const [editorHeight, setEditorHeight] = React.useState(getEditorHeight);

    function getEditorHeight () {
        if (isFullscreen) {
            return Math.max(MIN_EDITOR_HEIGHT, window.innerHeight - FULLSCREEN_VIEWPORT_OFFSET);
        } else {
            return Math.max(MIN_EDITOR_HEIGHT, window.innerHeight - VIEWPORT_OFFSET);
        }
    }

    // Resize handler
    React.useEffect(() => {
        const handleResize = () => {
            const newHeight = getEditorHeight();
            setEditorHeight(newHeight);

            if (editorRef.current) {
                editorRef.current.setSize("100%", `${newHeight}px`);
            }
        };

        window.addEventListener("resize", handleResize);
        return () => window.removeEventListener("resize", handleResize);
    }, []);

    return (
        <CodeMirror
            value={code}
            options={{
                mode: "javascript",
                theme: "default",
                lineNumbers: true
            }}
            editorDidMount={(editor) => {
                editorRef.current = editor;
                editor.setSize("100%", `${editorHeight}px`);
            }}
            onBeforeChange={(editor, data, value) => {
                setCode(value);
            }}
            onChange={(editor, data, value) => {
                setLocalDbValue("advancedSettings", value);
                updateSettingsChanges();
            }}
        />
    );
};

export const FlexCenterEnd = (props) => {
    return (
        <Flex
            justifyContent={{ default: "justifyContentFlexEnd" }}
            alignItems={{ default: "alignItemsCenter" }}
            style={{ height: "100%" }}
        >
            <FlexItem>
                {props.children}
            </FlexItem>
        </Flex>
    );
};

export const NODE_RED_STATES = {
    UP: "up",
    DOWN: "down",
    STARTING: "starting"
};

/**
 * A React hook to monitor and manage the state of a Node-RED instance. It listens for changes
 * in the state of the "nodered.service" systemd unit and checks if Node-RED is active, inactive,
 * or starting. The hook dynamically updates the connection URL and manages state polling using
 * curl requests.
 *
 * @param {string} [host="localhost"] - The hostname or IP address where Node-RED is running. Defaults to "localhost".
 * @param {number} [intervalMs=500] - The interval in milliseconds for polling Node-RED's status via curl requests. Defaults to 500ms.
 * @return {object} - Returns an object containing the state of the Node-RED instance.
 * @return {string|null} service return.nodeRedState - The current state of Node-RED, which may be one of
 * the predefined values (e.g., "STARTING", "UP", "DOWN") or null if the state is not yet determined.
 */

let curlTimer;

export function useNodeRedState (host = "localhost", intervalMs = 500) {
    const [nodeRedState, setNodeRedState] = React.useState(null);
    React.useEffect(() => {
        let lastState = null;
        subscribeToUnitChanges(NODE_RED_SERVICE + ".service", (detail) => {
            if (lastState === detail.ActiveState) return;
            lastState = detail.ActiveState;
            if (detail.ActiveState === "active") {
                setNodeRedState(NODE_RED_STATES.STARTING);
                startCurl();
            } else if (detail.ActiveState === "deactivating" || detail.ActiveState === "activating") {
                setNodeRedState(NODE_RED_STATES.STARTING);
            } else if (detail.ActiveState === "inactive" || detail.ActiveState === "failed") {
                stopCurl();
                setNodeRedState(NODE_RED_STATES.DOWN);
            }
        });
    }, []);

    const updateUrl = async () => {
        const pid = await getServicePid("nodered");
        const port = await getBindPorts(pid) || [1881];
        return `${host}:${port[0]}`;
    };

    const runCurl = async () => {
        const url = await updateUrl();
        cockpit.spawn(["curl", "--insecure", "--fail", url]).then(() => {
            setNodeRedState(NODE_RED_STATES.UP);
            clearInterval(curlTimer);
        });
    };

    function startCurl () {
        curlTimer = setInterval(runCurl, intervalMs);
    }

    function stopCurl () {
        clearInterval(curlTimer);
    }

    return { nodeRedState };
}

export const REVPI_NODES_STATES = {
    UP: "up",
    DOWN: "down",
    LOADING: "loading"
};

export function useRevpiNodesState () {
    const [nodesState, setNodesState] = React.useState(null);
    React.useEffect(() => {
        let lastState = null;
        subscribeToUnitChanges(NODES_SERVER_SERVICE + ".service", (detail) => {
            if (lastState === detail.ActiveState) return;
            lastState = detail.ActiveState;
            if (detail.ActiveState === "active") {
                setNodesState(REVPI_NODES_STATES.UP);
            } else if (detail.ActiveState === "deactivating" || detail.ActiveState === "activating") {
                setNodesState(REVPI_NODES_STATES.LOADING);
            } else {
                setNodesState(REVPI_NODES_STATES.DOWN);
            }
        });
    }, []);

    return { nodesState };
}

export const FullscreenToggle = ({
    isFullscreen,
    onToggle,
    style
}) => {
    return (
        <Tooltip content={isFullscreen ? _("Exit full screen") : _("Enter full screen")}>
            <Button
                variant='plain'
                aria-label={isFullscreen ? "Exit full screen" : "Enter full screen"}
                onClick={onToggle}
                style={style}
            >
                {isFullscreen ? <CompressIcon /> : <ExpandIcon />}
            </Button>
        </Tooltip>
    );
};

export const AdvancedSetting = ({ label, helpText, handler, isChecked, isLoading, children }) => {
    return (
        <GridItem span={12} style={{ marginBottom: "0.75rem" }}>
            <Flex direction={{ default: "column" }}>
                <FlexItem>
                    <SwitchWithSpinner
                        label={label}
                        isChecked={isChecked}
                        onChange={handler}
                        isLoading={isLoading}
                        isReversed
                        isAligned
                        helpText={
                            <TextContent>
                                <Text component='p'>
                                    {helpText}
                                </Text>
                            </TextContent>
                        }
                    />
                </FlexItem>
                <FlexItem>
                    {children}
                </FlexItem>
            </Flex>

        </GridItem>
    );
};
