// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import { useEffect, useState, useCallback, useRef } from "react";
import cockpit from "cockpit";
import { extractDbusValue } from "../../common/helper.js";

export function useFeatureStates (features) {
    const [availableFeatures, setAvailableFeatures] = useState({});
    const [featureStates, setFeatureStates] = useState({});
    const [isLoading, setIsLoading] = useState(true);
    const proxyCache = useRef(new Map());
    const unsubscribers = useRef([]);

    const getProxyForInterface = (interfaceName) => {
        if (!proxyCache.current.has(interfaceName)) {
            const client = cockpit.dbus("com.revolutionpi.middleware1", {
                superuser: "try",
                bus: "system"
            });
            const proxy = client.proxy(interfaceName, "/com/revolutionpi/middleware1");
            proxyCache.current.set(interfaceName, proxy);
        }
        return proxyCache.current.get(interfaceName);
    };

    const getProxyForFeature = (feature) => {
        const iface = feature.interface || "com.revolutionpi.middleware1.RevpiConfig";
        return getProxyForInterface(iface);
    };

    useEffect(() => {
        const subscribeToSignal = (interfaceName, member, handler) => {
            const client = cockpit.dbus("com.revolutionpi.middleware1", {
                superuser: "try",
                bus: "system"
            });

            const subscription = client.subscribe(
                {
                    interface: interfaceName,
                    path: "/com/revolutionpi/middleware1",
                    member
                },
                (_path, _iface, _member, args) => {
                    handler(...args.map(extractDbusValue));
                }
            );

            unsubscribers.current.push(() => subscription?.remove?.());
        };

        const fetchFeatureAvailabilityAndStatus = async (feature) => {
            const proxy = getProxyForFeature(feature);
            await proxy.wait();

            try {
                const availabilityResult = await proxy.call("GetAvailability", [feature.id]);
                const isAvailable = extractDbusValue(availabilityResult);

                setAvailableFeatures((prev) => ({
                    ...prev,
                    [feature.id]: isAvailable
                }));

                if (!isAvailable) return;

                const statusResult = await proxy.call("GetStatus", [feature.id]);
                const isEnabled = extractDbusValue(statusResult);

                setFeatureStates((prev) => ({
                    ...prev,
                    [feature.id]: isEnabled
                }));
            } catch (err) {
                console.warn(`Failed to fetch data for ${feature.id}`, err);
            }
        };

        const initialize = async () => {
            const proxies = new Set();
            features.forEach((feature) => {
                proxies.add(getProxyForFeature(feature));
            });

            await Promise.all(Array.from(proxies).map((proxy) => proxy.wait()));
            await Promise.all(features.map(fetchFeatureAvailabilityAndStatus));
            setIsLoading(false);

            const uniqueInterfaces = new Set(
                features.map((f) => f.interface || "com.revolutionpi.middleware1.RevpiConfig")
            );

            uniqueInterfaces.forEach((iface) => {
                subscribeToSignal(iface, "StatusChanged", (featureId, isEnabled) => {
                    setFeatureStates((prev) => ({
                        ...prev,
                        [featureId]: isEnabled
                    }));
                });

                subscribeToSignal(iface, "AvailabilityChanged", (featureId, isAvailable) => {
                    setAvailableFeatures((prev) => ({
                        ...prev,
                        [featureId]: isAvailable
                    }));
                });
            });
        };

        initialize();

        return () => {
            unsubscribers.current.forEach((unsubscribe) => unsubscribe());
            unsubscribers.current = [];
        };
    }, [features]);

    const toggle = useCallback(
        (id, enable) => {
            const feature = features.find((f) => f.id === id);
            if (!feature) return;

            const proxy = getProxyForFeature(feature);
            proxy.wait().then(() => {
                const method = enable ? "Enable" : "Disable";
                proxy.call(method, [id]);
            });
        },
        [features]
    );

    return {
        featureStates,
        availableFeatures,
        toggle,
        isLoading
    };
}
