// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";
import React, { useMemo } from "react";
import { ClipboardCopy } from "@patternfly/react-core";
import { useLoggedInUser } from "../../../pkg/lib/hooks.js";
import { getSshAddress } from "../../common/helper.js";
import { useCustomization } from "../../branding.js";

const _ = cockpit.gettext;

export function useFeatureList () {
    const user = useLoggedInUser();
    const { productName } = useCustomization();
    const sshAddress = useMemo(() => {
        if (!user) return "";
        return getSshAddress(user);
    }, [user]);
    return useMemo(() => [
        {
            id: "ntp",
            title: _("Set clock from NTP-Server"),
            type: "config",
            helpText: (
                <div className='help-text'>
                    <p>{_("This setting synchronizes your system time with an NTP server for accuracy.")}</p>
                    <p><strong>{_("Why is this needed?")}</strong></p>
                    <ul>
                        <li>{_("Ensures accurate timestamps for logs and scheduled tasks.")}</li>
                        <li>{_("Prevents time drift on devices without a real-time clock.")}</li>
                    </ul>
                    <p>
                        <strong>{_("Requirements")}:</strong> {_("An active internet connection is required to sync with NTP servers.")}
                    </p>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.SoftwareServices"
        },
        {
            id: "ssh",
            title: _("SSH daemon to enable remote login"),
            type: "config",
            helpText: (
                <div className='help-text'>
                    <p>{_("Enabling SSH allows remote terminal access to this device over a network.")}</p>
                    <p><strong>{_("Security Considerations")}:</strong></p>
                    <ul>
                        <li>{_("Use strong passwords or SSH key-based authentication.")}</li>
                        <li>{_("Disable SSH if remote access is not required.")}</li>
                    </ul>
                    <p><strong>{_("Usage")}:</strong> {_("To connect via SSH, use")}: </p>
                    <ClipboardCopy
                        isReadOnly
                        style={{ marginTop: "0.5rem" }}
                    >
                        ssh {sshAddress}
                    </ClipboardCopy>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.SoftwareServices"
        },
        {
            id: "gui",
            title: _("Boot to GUI"),
            type: "config",
            rebootRequired: true,
            helpText: (
                <div className='help-text'>
                    <p>{_("When enabled, the system boots into a graphical user interface (GUI) instead of the command line.")}</p>
                    <p><strong>{_("Considerations")}:</strong></p>
                    <ul>
                        <li>{_("GUI mode uses more RAM and processing power.")}</li>
                        <li>{_("Recommended for users unfamiliar with command-line operations.")}</li>
                    </ul>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.RevpiConfig"
        },
        {
            id: "bluetooth",
            title: _("BT"),
            type: "config",
            rebootRequired: true,
            helpText: (
                <div className='help-text'>
                    <p>{_("Enable BT for using Bluetooth peripherals such as keyboards or audio devices.")}</p>
                    <p>{_("Disable BT to reduce power consumption and improve security.")}</p>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.RevpiConfig"
        },
        {
            id: "external-antenna",
            title: _("Select antenna"),
            type: "config",
            rebootRequired: true,
            options: {
                enabled: {
                    title: _("External antenna (SMA)")
                },
                disabled: {
                    title: _("Internal antenna")
                }
            },
            helpText: (
                <div className='help-text'>
                    <p>{_("Switches between an internal and an external antenna for wireless connectivity.")}</p>
                    <div>
                        <p><strong>{_("Antenna Selection")}:</strong></p>
                        <ul>
                            <li>
                                <strong>{_("External antenna (SMA)")}:</strong>{" "}
                                {_("Provides better signal reception in challenging environments.")}
                            </li>
                            <li>
                                <strong>{_("Internal antenna")}:</strong>{" "}
                                {_("Uses the built-in antenna, reducing external hardware needs.")}
                            </li>
                        </ul>
                    </div>
                </div>

            ),
            interface: "com.revolutionpi.middleware1.RevpiConfig"
        },
        {
            id: "revpipyload",
            title: "RevPiPyLoad",
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{cockpit.format(_("RevPiPyLoad is a Python-based runtime environment for $0."), productName)}</p>
                    <p><strong>{_("Functions")}:</strong></p>
                    <ul>
                        <li>{_("Manages real-time industrial automation tasks.")}</li>
                        <li>{_("Runs user-defined Python scripts.")}</li>
                    </ul>
                    <p>{_("Ensure your scripts are compatible with RevPiPyLoad before enabling.")}</p>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.SoftwareServices"
        },
        {
            id: "avahi",
            title: _("Avahi (ZeroConf)"),
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{_("Avahi enables ZeroConf networking, allowing easy discovery of network services.")}</p>
                    <p><strong>{_("Examples of Use")}:</strong></p>
                    <ul>
                        <li>{_("Accessing the device using `hostname.local` instead of an IP address.")}</li>
                        <li>{_("Automatically discovering printers and file shares.")}</li>
                    </ul>
                    <p><strong>{_("Important")}:</strong></p>
                    {_("You cannot disable Avahi while connected to Cockpit via an Avahi address (.local). If you need to disable it, please use the command line or connect using the device's IP address instead.")}
                </div>
            ),
            hoverText: {
                isDisabled: _("You cannot disable Avahi while connected to Cockpit via an Avahi address (.local). If you need to disable it, please use the command line or connect using the device's IP address instead.")
            },
            interface: "com.revolutionpi.middleware1.SoftwareServices"
        },
        {
            id: "swapfile",
            title: _("Swap / page file (recommended for GUI)"),
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{_("Enables swap memory, which acts as virtual RAM when physical memory is low.")}</p>
                    <p><strong>{_("Impact")}:</strong></p>
                    <ul>
                        <li>{_("Prevents system crashes when RAM is exhausted.")}</li>
                        <li>{_("Can slow performance since swap is stored on disk.")}</li>
                    </ul>
                </div>
            ),
            interface: "com.revolutionpi.middleware1.RevpiConfig"
        }
    ], [sshAddress]);
}
