// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import cockpit from "cockpit";
import {
    Flex,
    FlexItem,
    MenuToggle,
    Select,
    SelectList,
    SelectOption,
    Spinner,
    Switch,
    TextInputGroup,
    TextInputGroupMain,
    TextInputGroupUtilities,
    ToggleGroup,
    ToggleGroupItem,
    Tooltip
} from "@patternfly/react-core";

import { ErrorIcon, HelpText } from "../common/ui-components.jsx";
import { Button } from "@patternfly/react-core/dist/esm/components/Button/index.js";
import { TimesIcon } from "@patternfly/react-icons";
import { getAllCountryCodes, getCurrentCountryCode, setNewCountryCode } from "../common/helper.js";

const _ = cockpit.gettext;

/**
 * Renders a switch UI element with contextual tooltips and status handling.
 * Used for enabling/disabling simple boolean features.
 *
 * @param {Object} props - Props for the component.
 * @param {Object} props.item - Object describing the switch state.
 * @param {Function} props.switchHandler - Callback to toggle state.
 */
export const RevPiSwitch = (props) => {
    const { id, isEnabled, isLoading } = props.item;
    const isDisabled = isLoading || props.item.isDisabled || props.disabled;

    return (
        <RevPiFlexBase {...props}>
            <HoverToolTip item={props.item}>
                <Switch
                    hasCheckIcon
                    id={id}
                    label={_("Enabled")}
                    labelOff={_("Disabled")}
                    isChecked={isEnabled}
                    isDisabled={isDisabled}
                    onChange={() => {
                        props.switchHandler(id, isEnabled, isLoading);
                    }}
                />
            </HoverToolTip>
        </RevPiFlexBase>
    );
};

/**
 * Renders a two-option toggle (e.g., On/Off) with labels.
 * Uses PatternFly ToggleGroupItem.
 *
 * @param {Object} props - Props for the component.
 * @param {Object} props.item - Object describing the toggle state.
 * @param {Function} props.switchHandler - Callback to toggle state.
 */
export const RevPiToggle = (props) => {
    const {
        id,
        isEnabled,
        isLoading,
        options = []
    } = props.item;

    const {
        enabled,
        disabled
    } = options;

    return (
        <RevPiFlexBase {...props}>
            <ToggleGroup>
                <>
                    <ToggleGroupItem
                        text={enabled.title}
                        isSelected={isEnabled === true}
                        onChange={() => props.switchHandler(id, false, isLoading)}
                    />
                    <ToggleGroupItem
                        text={disabled.title}
                        isSelected={isEnabled === false}
                        onChange={() => props.switchHandler(id, true, isLoading)}
                    />
                </>
            </ToggleGroup>
        </RevPiFlexBase>
    );
};

/**
 * Base layout wrapper for feature control rows.
 * Handles titles, help text, status indicators, and layout.
 *
 * @param {Object} props
 * @param {Object} props.item - Contains feature data (title, status, etc.)
 */
export function RevPiFlexBase (props) {
    const {
        title,
        helpText,
        hasError,
        isAvailable,
        isLoading,
        rebootRequired
    } = props.item;

    if (isAvailable === false) {
        return <></>;
    }

    return (
        <Flex
            style={{ minHeight: "2lh" }}
            direction={{ default: "row" }}
            alignItems={{ default: "alignItemsCenter" }}
        >
            <FlexItem grow={{ default: "grow" }}>
                <Flex direction={{ default: "row" }}>
                    <Flex direction={{ default: "column" }} gap={{ default: "gapNone" }} className='no-margin'>
                        <FlexItem>{title}</FlexItem>
                        {rebootRequired && <FlexItem className='card-title-label'>{_("reboot required")}</FlexItem>}
                    </Flex>
                    {helpText &&
                        <Flex>
                            <HelpText>
                                {helpText}
                            </HelpText>
                        </Flex>}
                </Flex>
            </FlexItem>

            <FlexItem>
                {isLoading && <Spinner size='lg' />}
                {hasError && !isLoading && <ErrorIcon errorText={_("Error")} />}
            </FlexItem>

            <FlexItem>
                {props.children}
            </FlexItem>
        </Flex>
    );
}

/**
 * Wraps a component with a conditional tooltip based on feature state.
 *
 * @param {Object} item - Item containing tooltip texts.
 * @param {React.ReactNode} children - The element to wrap with a tooltip.
 */
export const HoverToolTip = ({ item, children }) => {
    const { hoverText, isDisabled } = item;
    const toolTipContent = isDisabled
        ? hoverText?.isDisabled
        : hoverText?.isEnabled;
    if (!toolTipContent) return children;
    return (
        <Tooltip content={<div>{toolTipContent}</div>}>
            <span style={{ display: "inline-block" }}>{children}</span>
        </Tooltip>
    );
};
/**
 * Component for selecting and setting the WLAN country code.
 *
 * This component allows users to select a WLAN country code from a dropdown list.
 * It ensures compliance with regional regulations by letting users set and update
 * the WLAN country code accordingly. It displays a loading indicator while fetching
 * data or setting the new country code and can display help text about the importance
 * of the WLAN country code. If there are any errors during data fetching or updates,
 * it handles and displays error states.
 *
 * Properties:
 * @prop {Function} setIsWlanCountrySet - A callback function to update the parent
 * component's state or context when a WLAN country code is successfully set.
 */
export const WlanCountrySelect = ({ setIsWlanCountrySet }) => {
    const [isLoading, setIsLoading] = React.useState(true);
    const [hasError, setHasError] = React.useState(false);
    const [allCountryCodes, setAllCountryCodes] = React.useState([]);
    const [currentCountryCode, setCurrentCountryCode] = React.useState("");

    React.useEffect(() => {
        loadData();
    }, []);

    const loadData = async () => {
        setIsLoading(true);
        try {
            const allCodes = await getAllCountryCodes();
            const currentCode = await getCurrentCountryCode();
            setAllCountryCodes(allCodes);
            setCurrentCountryCode(currentCode);
            if (currentCode !== "") {
                setIsWlanCountrySet(true);
            }
        } catch (e) {
            console.error(e);
            setHasError(true);
        } finally {
            setIsLoading(false);
        }
    };

    const handleWlanCountrySelect = async (dropdownSelection) => {
        setIsLoading(true);
        try {
            await setNewCountryCode(dropdownSelection);
            setCurrentCountryCode(dropdownSelection);
            setIsWlanCountrySet(true);
        } catch (e) {
            console.error("Error while handling new wlan country", e);
            setHasError(true);
        } finally {
            setIsLoading(false);
        }
    };

    const getHelpText = () => (
        <div className='help-text'>
            <p>{_("The WLAN country code ensures that the system complies with regional regulations and only uses approved frequencies and transmission powers.")}</p>
        </div>
    );

    return (
        <RevPiFlexBase
            item={{
                title: _("WLAN country code"),
                rebootRequired: false,
                isLoading,
                hasError,
                helpText: getHelpText()
            }}
        >
            {allCountryCodes.length > 0 && (
                <SearchSelect
                    options={allCountryCodes}
                    status={currentCountryCode}
                    handleSelect={handleWlanCountrySelect}
                    placeholder={_("Select WLAN country code")}
                />
            )}
        </RevPiFlexBase>
    );
};
/**
 * SearchSelect is a React functional component designed for creating an accessible, customizable type-ahead search
 * and select dropdown. It allows users to filter through a list of options by typing, navigate options via keyboard,
 * and make selections efficiently. The component provides clear interaction states and can handle custom logic defined
 * by the parent component.
 *
 * Props:
 * @param {Object} props - The properties passed to the component.
 * @param {Array<string>} props.options - An array of string options to display in the dropdown. Each string should follow the format that includes a country code and name.
 * @param {string} props.status - The current status or pre-selected option's value. The component uses this to initialize the selection.
 * @param {Function} props.handleSelect - A callback function invoked when the user selects an option. The function receives the selected value as an argument.
 */
export const SearchSelect = (props) => {
    const {
        options,
        status = "",
        handleSelect,
        placeholder = _("Select..."),
        isDisabled = false
    } = props;

    const extractCode = (option) => option.split(" ")[0];
    const getFullString = (code) => options.find(c => c.includes(code));

    // initial setup
    const initialSelectOptions = options.map(e => ({
        value: extractCode(e),
        children: e
    }));
    const initialSelection = options.find(o => extractCode(o) === status.trim()) || "";

    // state
    const [isOpen, setIsOpen] = React.useState(false);
    const [selected, setSelected] = React.useState(initialSelection);
    const [inputValue, setInputValue] = React.useState(initialSelection);
    const [filterValue, setFilterValue] = React.useState("");
    const [selectOptions, setSelectOptions] = React.useState(initialSelectOptions);
    const [focusedItemIndex, setFocusedItemIndex] = React.useState(null);
    const textInputRef = React.useRef();

    React.useEffect(() => {
        let newOptions = initialSelectOptions;
        if (filterValue) {
            newOptions = initialSelectOptions.filter(opt =>
                String(opt.children).toLowerCase().includes(filterValue.toLowerCase())
            );
            if (!newOptions.length) {
                newOptions = [{
                    isDisabled: true,
                    children: cockpit.format(_("No results found for \"$0\""), filterValue),
                    value: "no-results"
                }];
            }
            if (!isOpen && !isDisabled) setIsOpen(true);
        }
        setSelectOptions(newOptions);
        setFocusedItemIndex(null);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [filterValue]);

    const onToggleClick = () => {
        if (isDisabled) return;
        setIsOpen(!isOpen);
    };

    const handleSelection = (value) => {
        if (isDisabled) return;
        if (options.find(o => o.includes(value)) && value.trim() !== "") {
            const full = getFullString(value);
            setInputValue(full);
            setSelected(full);
            handleSelect?.(value);
        }
    };

    const onSelect = (_event, value) => {
        if (isDisabled) return;
        if (value && value !== "no-results") {
            setFilterValue("");
            handleSelection(value);
        }
        setIsOpen(false);
        setFocusedItemIndex(null);
    };

    const onTextInputChange = (_event, value) => {
        if (isDisabled) return;
        setInputValue(value);
        setFilterValue(value);
    };

    const handleMenuArrowKeys = (key) => {
        if (isDisabled) return;
        let indexToFocus;
        if (isOpen) {
            if (key === "ArrowUp") {
                indexToFocus =
                    focusedItemIndex === null || focusedItemIndex === 0
                        ? selectOptions.length - 1
                        : focusedItemIndex - 1;
            }
            if (key === "ArrowDown") {
                indexToFocus =
                    focusedItemIndex === null || focusedItemIndex === selectOptions.length - 1
                        ? 0
                        : focusedItemIndex + 1;
            }
            setFocusedItemIndex(indexToFocus);
        }
    };

    const onInputKeyDown = (event) => {
        if (isDisabled) return;
        const enabled = selectOptions.filter(o => !o.isDisabled);
        const [first] = enabled;
        const focused = focusedItemIndex !== null ? enabled[focusedItemIndex] : first;

        switch (event.key) {
        case "Enter":
            if (isOpen && focused && focused.value !== "no-results") {
                setInputValue(String(focused.children));
                setFilterValue("");
                handleSelection(String(focused.value));
            }
            setIsOpen(prev => !prev);
            setFocusedItemIndex(null);
            break;
        case "Tab":
        case "Escape":
            setIsOpen(false);
            break;
        case "ArrowUp":
        case "ArrowDown":
            event.preventDefault();
            handleMenuArrowKeys(event.key);
            break;
        }
    };

    const toggle = (toggleRef) => (
        <MenuToggle
            ref={toggleRef}
            variant='typeahead'
            onClick={onToggleClick}
            isExpanded={!isDisabled && isOpen}
            isFullWidth
            isDisabled={isDisabled}
        >
            <TextInputGroup isPlain>
                <TextInputGroupMain
                    value={inputValue}
                    onClick={onToggleClick}
                    onChange={onTextInputChange}
                    onKeyDown={onInputKeyDown}
                    innerRef={textInputRef}
                    placeholder={placeholder}
                    autoComplete='off'
                />
                <TextInputGroupUtilities>
                    {!!inputValue && !isDisabled && (
                        <Button
                            variant='plain'
                            onClick={() => {
                                handleSelection("");
                                setInputValue("");
                                setFilterValue("");
                                textInputRef?.current?.focus();
                            }}
                        >
                            <TimesIcon />
                        </Button>
                    )}
                </TextInputGroupUtilities>
            </TextInputGroup>
        </MenuToggle>
    );

    return (
        <Select
            id='typeahead-select'
            isOpen={!isDisabled && isOpen}
            selected={selected}
            onSelect={onSelect}
            onOpenChange={() => setIsOpen(false)}
            toggle={toggle}
        >
            <SelectList
                id='select-typeahead-listbox'
                style={{ maxHeight: "250px", overflow: "auto" }}
            >
                {selectOptions.map((option, index) => (
                    <SelectOption
                        key={option.value || option.children}
                        isFocused={focusedItemIndex === index}
                        className={option.className}
                        id={`select-typeahead-${String(option.value).replace(" ", "-")}`}
                        {...option}
                        ref={null}
                    />
                ))}
            </SelectList>
        </Select>
    );
};
