// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import cockpit from "cockpit";
import { ExternalLinkAltIcon, IntegrationIcon, PowerOffIcon, RedoIcon } from "@patternfly/react-icons";
import {
    Card,
    CardBody,
    CardTitle,
    GridItem,
    Grid,
    Flex,
    FlexItem,
    Title,
    DescriptionListGroup,
    DescriptionListTerm,
    DescriptionListDescription,
    DescriptionList,
    ClipboardCopy,
    Alert,
    ListItem,
    List,
    Bullseye,
    Spinner,
    Button,
    Stack,
    StackItem,
    CodeBlock,
    CodeBlockCode
} from "@patternfly/react-core";
import {
    ErrorMessage,
    MacAddressInput,
    RevPiSelect,
    SerialNumberForm,
    UiCardTitle,
    UiDivider
} from "../../common/ui-components.jsx";
import { reboot } from "../../common/helper.js";
import { HelperText, HelperTextItem } from "@patternfly/react-core/dist/esm/components/HelperText";

const _ = cockpit.gettext;

export function FactoryReset ({
    hasFactoryReset,
    setFactoryResetError,
    hasHatEeprom,
    resetOccurredThisBoot,
    hostname,
    password,
    runFactoryReset,
    factoryResetError,
    setHasFactoryReset,
    isFactoryResetRunning,
    children // fallback content when no factory-reset UI is needed
}) {
    if (factoryResetError) {
        return (
            <ErrorMessage error={factoryResetError}>
                <Flex
                    justifyContent={{ default: "justifyContentCenter" }}
                    spaceItems={{ default: "spaceItemsMd" }}
                    alignItems={{ default: "alignItemsCenter" }}
                    style={{ marginTop: 16 }}
                >
                    <FlexItem>
                        <Button
                            variant='primary'
                            onClick={() => {
                                setFactoryResetError(null);
                                setHasFactoryReset(false);
                                runFactoryReset();
                            }}
                            icon={<RedoIcon />}
                        >
                            {_("Try setup again")}
                        </Button>
                    </FlexItem>

                    <FlexItem>
                        <Button
                            variant='secondary'
                            onClick={reboot}
                            icon={<PowerOffIcon />}
                        >
                            {_("Reboot and retry")}
                        </Button>
                    </FlexItem>
                </Flex>
            </ErrorMessage>
        );
    }

    if (isFactoryResetRunning && !hasFactoryReset) {
        return (
            <Bullseye>
                <div style={{ textAlign: "center" }}>
                    <Spinner size='xl' />
                    <Title headingLevel='h2' size='lg'>
                        {_("Performing initial device setup...")}
                    </Title>
                    <p>{_("Please do not power off your device.")}</p>
                </div>
            </Bullseye>
        );
    }

    if (!hasFactoryReset && !isFactoryResetRunning && !hasHatEeprom) {
        return <FactoryResetCard onRunFactoryReset={runFactoryReset} />;
    }

    if (resetOccurredThisBoot && hasFactoryReset) {
        return <FactoryResetResultCard hostname={hostname} password={password} hasHasEeprom={hasHatEeprom} />;
    }

    // ✅ Nothing special to show → render the rest of the app
    return <>{children}</>;
}

function FactoryResetCard ({ onRunFactoryReset }) {
    // device type selection state
    const overlayOptions = {
        compact: "RevPi Compact",
        connect: "RevPi Connect(+) / Connect S",
        "connect-se": "RevPi Connect SE",
        core: "RevPi Core 3(+) / Core S",
        flat: "RevPi Flat"
    };

    const [, setTypeOpen] = React.useState(false);
    // User Input data if no Hat Eeprom is present
    const [macAddress, setMacAddress] = React.useState("");
    const [deviceOverlay, setDeviceOverlay] = React.useState("");
    const [serialNumber, setSerialNumber] = React.useState("");

    const onOverlaySelect = (value) => {
        setDeviceOverlay(value);
        setTypeOpen(false);
    };

    return (
        <Card isFlat>
            <CardTitle
                style={{
                    backgroundColor: "var(--pf-v5-global--palette--blue-50)",
                    padding: "1rem"
                }}
            >
                <UiCardTitle>
                    <IntegrationIcon style={{ marginRight: "0.75rem" }} />
                    {_("Initial Device Setup")}
                </UiCardTitle>
            </CardTitle>

            <CardBody>
                {/* Intro copy */}
                <div style={{ marginBottom: "0.75rem" }}>
                    {_("Your device is being prepared with defaults. Please provide the details below to complete the setup.")}
                </div>

                {/* Section: Device type */}
                <Title headingLevel='h3' size='lg' style={{ marginBottom: "0.25rem" }}>
                    {_("Select device type")}
                </Title>

                <RevPiSelect
                    id='factory-reset-select-device-type'
                    value={deviceOverlay}
                    onChange={onOverlaySelect}
                    options={overlayOptions}
                    placeholder={_("Choose the matching RevPi model")}
                />

                <UiDivider style={{ marginBottom: "1rem" }} />

                {/* Section: Inputs (MAC + Serial) */}
                <Title headingLevel='h3' size='lg' style={{ marginBottom: "0.25rem" }}>
                    {_("Enter device identifiers")}
                </Title>

                <div style={{ marginBottom: "0.5rem" }}>
                    {_("Use the values printed on the front plate.")}
                </div>

                <Grid hasGutter>
                    <GridItem md={6} sm={12}>
                        <MacAddressInput
                            id='macAddressInput'
                            label={_("MAC Address")}
                            initial='C83E-A7'
                            groups={[4, 4, 4]} // XXXX-SSSS-TTTT
                            separator='-' // hyphen
                            // uppercase enforced in your latest MacAddressInput version
                            onChange={(normalized) => {
                                setMacAddress(normalized);
                            }}
                        />
                    </GridItem>

                    <GridItem md={6} sm={12}>
                        <SerialNumberForm
                            onChange={(value, valid) => setSerialNumber(value)}
                        /> {/* keep it ≤10 chars inside this component */}
                    </GridItem>
                </Grid>

                <Flex justifyContent={{ default: "justifyContentFlexEnd" }}>
                    <FlexItem>
                        <Button
                            onClick={() => onRunFactoryReset({
                                mac: macAddress,
                                serial: serialNumber,
                                overlay: deviceOverlay
                            })} variant='primary' isDisabled={!(macAddress.length === 12 && serialNumber && deviceOverlay)}
                        >
                            {_("Run initial setup")}
                        </Button>
                    </FlexItem>
                </Flex>
            </CardBody>
        </Card>
    );
}

function FactoryResetResultCard ({
    hostname,
    password,
    hasHasEeprom
}) {
    const avahiUrl = `https://${hostname.toLowerCase()}.local:41443`;
    return (
        <Alert
            isInline
            variant='success'
            title={_("Initial setup finished")}
            actionLinks={
                <Button
                    variant='primary'
                    icon={<RedoIcon />}
                    onClick={reboot}
                >
                    {_("Reboot now")}
                </Button>
            }
        >
            <div style={{ marginBottom: 8 }}>
                {_("Your device setup has completed successfully.")}
            </div>

            {/* Centered card container */}
            <div style={{
                display: "flex",
                justifyContent: "center",
                margin: "1rem 0"
            }}
            >
                <Card
                    isCompact
                    style={{
                        display: "inline-block",
                        padding: "0.5rem 1rem"
                    }}
                >
                    <CardBody
                        style={{
                            display: "flex",
                            justifyContent: "center"
                        }}
                    >
                        <DescriptionList
                            isCompact
                            columnModifier={{ default: "3Col" }}
                            style={{
                                textAlign: "center"
                            }}
                        >
                            <DescriptionListGroup>
                                <DescriptionListTerm>{_("Default user")}</DescriptionListTerm>
                                <DescriptionListDescription>
                                    <code
                                        style={{
                                            padding: "2px 6px",
                                            borderRadius: "4px"
                                        }}
                                    >
                                        pi
                                    </code>
                                </DescriptionListDescription>
                            </DescriptionListGroup>

                            {hostname && (
                                <DescriptionListGroup>
                                    <DescriptionListTerm>{_("Hostname")}</DescriptionListTerm>
                                    <DescriptionListDescription>
                                        <code
                                            style={{
                                                padding: "2px 6px",
                                                borderRadius: "4px"
                                            }}
                                        >
                                            {hostname}
                                        </code>
                                    </DescriptionListDescription>
                                </DescriptionListGroup>
                            )}

                            {password && (
                                <DescriptionListGroup>
                                    <DescriptionListTerm>{_("Password")}</DescriptionListTerm>
                                    <DescriptionListDescription>
                                        <ClipboardCopy
                                            isReadOnly
                                            hoverTip={_("Copy")}
                                            clickTip={_("Copied")}
                                            style={{
                                                width: "10rem",
                                                display: "inline-block",
                                                verticalAlign: "middle",
                                                margin: "0 auto"
                                            }}
                                        >
                                            {password}
                                        </ClipboardCopy>
                                    </DescriptionListDescription>
                                </DescriptionListGroup>
                            )}
                        </DescriptionList>
                    </CardBody>
                </Card>
            </div>

            <Stack hasGutter>
                <StackItem>
                    <Title headingLevel='h2' size='lg' style={{ fontWeight: 700 }}>
                        {_("Next steps")}
                    </Title>
                </StackItem>

                <StackItem>
                    <List isPlain>
                        <ListItem>
                            {_("Select “Reboot now” to apply all changes.")}
                        </ListItem>
                        <ListItem>
                            {_("After the reboot, sign in as")}{" "}
                            <CodeBlock isInline>
                                <CodeBlockCode>pi</CodeBlockCode>
                            </CodeBlock>{" "}
                            {_("using the password shown above.")}
                        </ListItem>
                        <ListItem>
                            {_("If Avahi (mDNS) is enabled, you can access your device in a browser at")}{" "}
                            <Button
                                component='a'
                                target='_blank'
                                rel='noopener noreferrer'
                                href={avahiUrl}
                                variant='link'
                                icon={<ExternalLinkAltIcon />}
                                iconPosition='end'
                                style={{ paddingLeft: 4 }}
                            >
                                {avahiUrl}
                            </Button>
                            <HelperText style={{ marginTop: 4 }}>
                                <HelperTextItem isDynamic>
                                    {_("Note: mDNS (.local) requires Avahi on the device.")}
                                </HelperTextItem>
                                {!hasHasEeprom &&
                                    (
                                        <HelperTextItem isDynamic>
                                            {_("Note: Your IP address may change after reboot if DHCP is enabled.")}
                                        </HelperTextItem>
                                    )}
                            </HelperText>
                        </ListItem>
                    </List>
                </StackItem>

                <StackItem>
                    <Alert
                        isInline
                        variant='warning'
                        title={_("For security reasons, it is recommended to change the default password after the reboot.")}
                    />
                </StackItem>

            </Stack>
        </Alert>
    );
}
