// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import {
    Button,
    Card,
    CardBody,
    CardTitle,
    Flex,
    FlexItem,
    Grid,
    GridItem,
    Text,
    TextContent,
    TextVariants
} from "@patternfly/react-core";
import React from "react";
import cockpit from "cockpit";
import { getHostName, getPortNumber, parseFilePath } from "../../common/helper.js";
import { useCustomization } from "../../branding.js";
import { ThIcon } from "@patternfly/react-icons";
import { DISPLAY_TYPE_COLORS, DISPLAY_TYPES, HelpText, UiCardTitle, UiDivider } from "../../common/ui-components.jsx";
import { useAppConfig, useNoderedPackageNames, useNotifications } from "../../common/hooks.jsx";

const _ = cockpit.gettext;

export function AppCard () {
    const { addNotification } = useNotifications();
    const { productName } = useCustomization();
    const revpiNodeRedPackageNames = useNoderedPackageNames();

    const apps = [
        // pictory
        useAppConfig({
            name: "pictory",
            displayName: "PiCtory",
            packageNames: ["pictory", "pictory-apache"],
            displayType: DISPLAY_TYPES.OPEN,
            click: openPictoryHandler(addNotification),
            helpText: (
                <HelpText>
                    <div className='help-text'>
                        <p>
                            {cockpit.format(_("Manages the configuration file of a Revolution Pi system, i.e. for a $0 base module with expansion modules."), productName)}
                        </p>
                    </div>
                </HelpText>
            )
        }),
        // nodered
        useAppConfig({
            name: "nodered",
            displayName: "Node-RED",
            packageNames: revpiNodeRedPackageNames,
            displayType: DISPLAY_TYPES.CONFIGURE,
            click: () => { cockpit.jump("/revpi-nodered") },
            helpText: (
                <HelpText>
                    {_("A simple flow-based tool for connecting devices and automating workflows.")}
                </HelpText>

            )
        })
    ];

    const handleAppAction = (app) => {
        if (!app.installer.isInstalled) {
            app.installer.handleInstall?.();
        } else {
            app.click?.();
        }
    };

    return (
        <Card isFullHeight name='app-list'>
            <CardTitle>
                <UiCardTitle>
                    <ThIcon style={{ marginRight: "1rem" }} />
                    {cockpit.format("$0 Apps", productName)}
                </UiCardTitle>
                <UiDivider />
            </CardTitle>
            <CardBody>
                <Grid hasGutter>
                    {apps.map((app) => (
                        <GridItem key={app.name} span={12}>
                            <div className='app-card'>
                                <Flex alignItems={{ default: "alignItemsCenter" }} justifyContent={{ default: "justifyContentSpaceBetween" }}>
                                    <FlexItem>
                                        <TextContent>
                                            <Text component={TextVariants.h2} style={{ fontWeight: "bold" }}>
                                                {app.displayName}
                                                {app.helpText}
                                            </Text>
                                        </TextContent>
                                    </FlexItem>
                                    <FlexItem>
                                        <Button
                                            variant={
                                                app.installer.isInstalled
                                                    ? DISPLAY_TYPE_COLORS[app.displayType]
                                                    : DISPLAY_TYPE_COLORS.install
                                            }
                                            onClick={() => handleAppAction(app)}
                                            style={{ minWidth: "115px" }}
                                            isDisabled={app.installer.isInstalling}
                                        >
                                            <span style={{ fontWeight: "bold" }}>
                                                {app.installer.isInstalled ? _(app.displayType) : _(DISPLAY_TYPES.INSTALL)}
                                            </span>
                                        </Button>
                                    </FlexItem>
                                </Flex>
                            </div>
                        </GridItem>
                    ))}

                </Grid>
            </CardBody>
        </Card>
    );
}

const openPictoryHandler = (addNotification) => {
    return async () => {
        try {
            const token = await createSsoToken();
            const defaultPort = getPortNumber();
            const pictoryPortNumber = defaultPort === 9090 ? 41443 : defaultPort;
            const url = `https://${getHostName()}:${pictoryPortNumber}/pictory/php/sso_login.php?sso_token=${token}`;
            window.open(url, "_blank");
        } catch (error) {
            addNotification(_("Could not launch PiCtory due to a login token error."), "", "danger");
            console.error("Error while creating SSO token", error);
        }
    };
};

// Function to create an SSO token by checking multiple script locations
const createSsoToken = async () => {
    const filePath = await parseFilePath("scripts/create_sso_token.sh");
    try {
        return await cockpit.spawn(filePath, { superuser: "require" });
    } catch (error) {
        console.log(`No script found at '${filePath}'.`, error);
    }
};
