// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";
import React from "react";
import {
    Page,
    PageSection,
    PageSectionVariants,
    Grid,
    GridItem,
    Alert
} from "@patternfly/react-core";
import {
    RevPiSwitch,
    RevPiToggle
} from "./revpi-config.components.jsx";
import { NetworkCard } from "./cards/network-card.jsx";
import { SystemConfigurationCard } from "./cards/system-configuration-card.jsx";
import { RunTimeCard } from "./cards/runtime-card.jsx";
import {
    getRevPiConfigStatus,
    hasLowMemory,
    toggleConfiguration
} from "./revpi-config/revpi-config-service.jsx";
import { useConfigData } from "./revpi-config/revpi-config-data.jsx";
import { installEventEmitter } from "./emitters.js";
import { AppCard } from "./cards/app-card.jsx";
import { AdminAccessButton, NotificationProvider, useFeatureInstaller, useNotifications } from "../common/hooks.jsx";
import { ErrorMessage, Notifications } from "../common/ui-components.jsx";
import { RegionalSettingsCard } from "./cards/regional-card.jsx";
import { FactoryReset } from "./cards/factory-reset-card.jsx";
import { detectFactoryReset, detectHatEeprom, doFactoryReset } from "../common/helper.js";

const _ = cockpit.gettext;

export const ApplicationRoot = ({ children }) => (
    <NotificationProvider>
        <Page>
            <PageSection variant={PageSectionVariants.default}>
                <Notifications />
                {children}
            </PageSection>
        </Page>
    </NotificationProvider>
);

export const Application = () => {
    // Global loading and error state
    const [isLoading, setIsLoading] = React.useState(true);
    const [features, setFeatures] = React.useState(new Map());
    const [error, setError] = React.useState(null);

    // First start after flashing operation
    const [isFactoryResetRunning, setIsFactoryResetRunning] = React.useState(false);
    const [hasHatEeprom, setHasHatEeprom] = React.useState(true);
    const [hasFactoryReset, setHasFactoryReset] = React.useState(true);
    const [resetOccurredThisBoot, setResetOccurredThisBoot] = React.useState(false);
    const [factoryResetError, setFactoryResetError] = React.useState(null);

    // Data from factory reset
    const [password, setPassword] = React.useState(null);
    const [hostname, setHostname] = React.useState(null);

    const { addNotification } = useNotifications();
    const configData = useConfigData();

    const runFactoryReset = async (opts) => {
        try {
            setIsFactoryResetRunning(true);
            await new Promise(resolve => setTimeout(resolve, 2000));
            // require overlay/serial/mac together; otherwise let backend auto-detect
            const needsManual = !!opts && ["overlay", "serial", "mac"].every(k => !!opts[k]);
            const payload = needsManual ? opts : undefined;

            const { password, hostname } = await doFactoryReset(payload);
            setPassword(password);
            setHostname(hostname);
            setHasFactoryReset(true);
        } catch (err) {
            console.error("Factory reset failed:", err);
            setFactoryResetError(err.message || "unknown-error");
        }
    };

    React.useEffect(() => {
        async function fetchFactoryReset () {
            const factoryResetExists = await detectFactoryReset();
            setHasFactoryReset(factoryResetExists);
            if (!factoryResetExists) {
                setResetOccurredThisBoot(true);
                const eepromExists = await detectHatEeprom();
                setHasHatEeprom(eepromExists);
                if (eepromExists) {
                    runFactoryReset();
                }
            }
        }

        fetchFactoryReset();
    }, []);

    // Track individual feature's loading state
    const updateFeatureLoadingState = React.useCallback((id, isLoading) => {
        setFeatures((prev) =>
            new Map(prev).set(id, { ...prev.get(id) || {}, isLoading: !isLoading })
        );
    }, []);

    // Track feature-level error
    const setFeatureError = React.useCallback((id) => {
        setFeatures((prev) =>
            new Map(prev).set(id, { ...prev.get(id) || {}, hasError: true })
        );
    }, []);

    // Fetches all features and updates state
    const loadAllFeatures = React.useCallback(async () => {
        try {
            const featureList = await getRevPiConfigStatus(configData);
            setFeatures(new Map(featureList.map((x) => [x.id, x])));
            setIsLoading(false);
        } catch (error) {
            console.error("Error fetching features:", error);
            setError(error.message || "unknown-error");
            setIsLoading(false);
        }
    }, [configData]);

    // Reload features when an installation is finished
    React.useEffect(() => {
        installEventEmitter.addEventListener(
            installEventEmitter.events.finish,
            loadAllFeatures
        );
        return () => {
            installEventEmitter.removeEventListener(
                installEventEmitter.events.finish,
                loadAllFeatures
            );
        };
    }, [loadAllFeatures]);

    // Reload features manually
    const refreshFeatures = React.useCallback(async () => {
        setError(null);
        setIsLoading(true);
        await loadAllFeatures();
    }, [loadAllFeatures]);

    // Handles enabling "swap" if device is low on memory and GUI is being enabled
    const handleGuiEnableLogic = React.useCallback(async () => {
        const lowMemory = await hasLowMemory();
        if (!lowMemory) return;

        const swapId = "dphys-swapfile";
        updateFeatureLoadingState(swapId, isLoading);

        addNotification(
            _("Swap / Page File Enabled for Better Performance"),
            cockpit.format(
                _("Your device has 1GB of RAM or less. To ensure smooth performance in GUI mode, \"$0\" has been automatically enabled. You can disable it if you prefer."),
                _("Swap / page file (recommended for GUI)")
            ),
            "info"
        );

        await toggleConfiguration(swapId, false);
    }, [updateFeatureLoadingState, addNotification, isLoading]);

    // Handles switch toggle and related logic
    const handleSwitchClick = React.useCallback(async (id, isEnabled, isLoading) => {
        updateFeatureLoadingState(id, isLoading);
        try {
            await toggleConfiguration(id, isEnabled);
            if (id === "gui" && !isEnabled) {
                await handleGuiEnableLogic();
            }
            await loadAllFeatures();
        } catch {
            updateFeatureLoadingState(id, !isLoading);
            setFeatureError(id);
        }
    }, [loadAllFeatures, updateFeatureLoadingState, setFeatureError, handleGuiEnableLogic]);

    // Utility to render switch component for a given feature ID
    const getRevPiSwitch = React.useCallback((id, props = {}) => {
        return <RevPiSwitch item={features.get(id)} switchHandler={handleSwitchClick} {...props} />;
    }, [features, handleSwitchClick]);

    // Utility to render toggle component for a given feature ID
    const getRevPiToggle = React.useCallback((id) => {
        return <RevPiToggle item={features.get(id)} switchHandler={handleSwitchClick} />;
    }, [features, handleSwitchClick]);

    // Check if Node-RED is installed but the cockpit plugin isn't
    const {
        isInstalled: isNodeRedInstalled
    } = useFeatureInstaller("Node-RED", ["revpi-nodered"]);

    const {
        isInstalled: isCockpitNodeRedInstalled
    } = useFeatureInstaller("Cockpit Node-RED Plugin", ["cockpit-revpi-nodered"]);

    const showNoderedPluginInfo = isNodeRedInstalled && !isCockpitNodeRedInstalled;

    // Permissions check on mount (required for feature access)
    React.useEffect(() => {
        const load = async () => {
            setError(null);
            setIsLoading(true);

            const permission = cockpit.permission({ admin: true });

            const handlePermissionChange = async () => {
                if (permission.allowed) {
                    await refreshFeatures();
                } else {
                    setError("access-denied");
                    setIsLoading(false);
                }
            };

            permission.addEventListener("changed", handlePermissionChange);

            await handlePermissionChange();

            return () => {
                permission.removeEventListener("changed", handlePermissionChange);
            };
        };

        load();
    }, [refreshFeatures]);

    return (
        <>
            {isLoading || error
                ? (
                    error === "access-denied"
                        ? (
                            <>
                                <AdminAccessButton />
                            </>
                        )
                        : (
                            <ErrorMessage loading={isLoading} error={error} />
                        )
                )
                : (

                    <FactoryReset
                        hasFactoryReset={hasFactoryReset}
                        setHasFactoryReset={setHasFactoryReset}
                        hasHatEeprom={hasHatEeprom}
                        resetOccurredThisBoot={resetOccurredThisBoot}
                        factoryResetError={factoryResetError}
                        setFactoryResetError={setFactoryResetError}
                        password={password}
                        hostname={hostname}
                        runFactoryReset={runFactoryReset}
                        isFactoryResetRunning={isFactoryResetRunning}
                    >
                        {/* Main application content */}
                        <Grid className='ct-system-overview' hasGutter>
                            {/* Node red plugin info alert */}
                            {showNoderedPluginInfo && (
                                <GridItem span={12}>
                                    <Alert
                                        variant='info'
                                        isInline
                                        title={_("Node-RED is installed, but the Cockpit integration is missing.")}
                                    >
                                        <p>
                                            {_(
                                                "We detected Node-RED on your system. For easier setup and configuration, we recommend installing the Cockpit Node-RED plugin. You can do this using the 'Install' button in the Apps list."
                                            )}
                                        </p>
                                    </Alert>
                                </GridItem>
                            )}

                            <GridItem xl={6} md={12}>
                                <AppCard />
                            </GridItem>
                            <GridItem xl={6} md={12}>
                                <RunTimeCard
                                    getRevPiSwitch={getRevPiSwitch}
                                    getRevPiToggle={getRevPiToggle}
                                />
                            </GridItem>
                            <GridItem xl={6} md={12}>
                                <SystemConfigurationCard
                                    getRevPiSwitch={getRevPiSwitch}
                                    getRevPiToggle={getRevPiToggle}
                                />
                            </GridItem>
                            <GridItem xl={6} md={12}>
                                <NetworkCard
                                    getRevPiSwitch={getRevPiSwitch}
                                    getRevPiToggle={getRevPiToggle}
                                />
                            </GridItem>
                            <GridItem xl={6} md={12}>
                                <RegionalSettingsCard />
                            </GridItem>
                        </Grid>
                    </FactoryReset>

                )}
        </>

    );
};
