// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";

// systemd D-Bus API names
const SYSTEMD_OBJECT_PATH = "/org/freedesktop/systemd1";
const SYSTEMD_INTERFACE = "org.freedesktop.systemd1";
const SYSTEMD_MANAGER_INTERFACE = "org.freedesktop.systemd1.Manager";
const SYSTEMD_UNIT_INTERFACE = "org.freedesktop.systemd1.Unit";
const SYSTEMD_SERVICE_INTERFACE = "org.freedesktop.systemd1.Service";
const DBUS_PROPERTIES_INTERFACE = "org.freedesktop.DBus.Properties";

const systemdClient = cockpit.dbus(SYSTEMD_INTERFACE, { superuser: "try" });
const systemdManager = systemdClient.proxy(
    SYSTEMD_MANAGER_INTERFACE,
    SYSTEMD_OBJECT_PATH
);

/**
 * Enables and starts a systemd service.
 *
 * @param {string} serviceName - The base name of the service (e.g., "nodered", not "nodered.service").
 * @returns {Promise} A promise that resolves when the service has been enabled and started.
 */
export const enableService = async (serviceName) => {
    const unitName = serviceName + ".service";
    try {
        await systemdManager.call("StartUnit", [unitName, "replace"]);
        // D-Bus call: EnableUnitFiles
        // in: array string files         - List of unit files to enable
        // in: boolean runtime           - Whether to only enable temporarily until reboot
        // in: boolean force            - Ignore various types of errors when enabling
        // out: boolean carries_install_info - Whether unit files carry install information
        // out: array(sss) changes      - Changes that were made
        const unitResponse = await systemdManager.call("EnableUnitFiles", [[unitName], false, false]);
        if (unitResponse[1].length > 0) {
            // reload systemd if any unit files were enabled
            reloadSystemd();
        }
        // StartUnit method takes:
        // - name (string): The name of the unit to start
        // - mode (string): The start mode (e.g. "replace", "fail", "isolate", etc.)
        // Returns:
        // - job (object): The job object path
    } catch (error) {
        console.log(`Error starting and enabling ${unitName}: ${error}`);
    }
};

/**
 * Disables and stops a systemd service.
 *
 * @param {string} serviceName - The base name of the service (without ".service").
 * @returns {Promise} A promise that resolves when the service has been disabled and stopped.
 */
export const disableService = async (serviceName) => {
    const unitName = serviceName + ".service";
    try {
        await systemdManager.call("StopUnit", [unitName, "replace"]);
        // D-Bus call: DisableUnitFiles
        // in: array string files        - List of unit files to disable
        // in: boolean runtime          - Whether to only disable temporarily until reboot
        // out: array(sss) changes     - Changes that were made
        const unitResponse = await systemdManager.call("DisableUnitFiles", [[unitName], false]);
        if (unitResponse.length > 0) {
            // reload systemd if any unit files were disabled
            reloadSystemd();
        }
    } catch (error) {
        console.log(`Error stopping and disabling ${unitName}: ${error}`);
    }
};

/**
 * Restarts a systemd service.
 *
 * @param {string} serviceName - The base name of the service (without ".service").
 * @returns {Promise} A promise that resolves when the service has been restarted.
 */
export const restartService = (serviceName) => {
    const unitName = serviceName + ".service";
    try {
        // D-Bus call: RestartUnit
        // in: string name             - The name of the unit to restart
        // in: string mode             - The restart mode ("replace", "fail", "isolate", "ignore-dependencies", "ignore-requirements")
        // out: object job             - The job object path
        return systemdManager.call("RestartUnit", [unitName, "replace"]);
    } catch (error) {
        console.log(`Error restarting ${unitName}: ${error}`);
    }
};

/**
 * Checks whether a systemd service is enabled (starts automatically on boot).
 *
 * @param {string} serviceName - The base name of the service (without ".service").
 * @returns {Promise<boolean>} A promise that resolves to true if enabled, false otherwise.
 */
export const isServiceEnabled = async (serviceName) => {
    const unitName = serviceName + ".service";
    const unitState = await getUnitProperty(unitName, "UnitFileState");
    return unitState === "enabled";
};

/**
 * Checks whether a systemd service is currently active (running).
 *
 * @param {string} serviceName - The base name of the service (without ".service").
 * @returns {Promise<boolean>} A promise that resolves to true if active, false otherwise.
 */
export const isServiceActive = async (serviceName) => {
    const unitName = serviceName + ".service";
    const activeState = await getUnitProperty(unitName, "ActiveState");
    return activeState === "active";
};

/**
 * Checks whether a systemd service is masked (prevented from starting).
 *
 * @param {string} serviceName - The base name of the service (without ".service").
 * @returns {Promise<boolean>} A promise that resolves to true if masked, false otherwise.
 */
export const isServiceMasked = async (serviceName) => {
    const unitName = serviceName + ".service";
    const unitState = await getUnitProperty(unitName, "UnitFileState");
    return unitState === "masked";
};

/**
 * Retrieves the main PID of a given systemd service.
 *
 * This function appends ".service" to the provided service name,
 * fetches the service's D-Bus properties, and extracts the main process ID.
 *
 * @param {string} serviceName - The base name of the service (e.g., "nodered").
 * @returns {Promise<number>} A promise that resolves to the main PID of the service, or `0` if the service is not running.
 *
 * @example
 * const pid = await getServicePid("nodered");
 * console.log("Node-RED is running with PID:", pid);
 */
export const getServicePid = async (serviceName) => {
    const unitName = serviceName + ".service";
    return await getServiceProperty(unitName, "MainPID");
};

/**
 * Retrieves a proxy object for accessing the properties of a specific systemd unit.
 *
 * This function loads the specified systemd unit and returns a proxy object
 * for interacting with its properties over D-Bus. The proxy is created using
 * the D-Bus Properties Interface.
 *
 * @param {string} unitName - The name of the systemd unit to load.
 * @returns {Promise<object>} A promise that resolves to a D-Bus proxy object for the unit's properties.
 * @throws {Error} Throws an error if the systemd unit cannot be loaded or the proxy cannot be created.
 */
const getPropertiesProxy = async (unitName) => {
    // D-Bus call: LoadUnit
    // in:  string name              - The name of the unit to load
    // out: object unit             - Object path of the loaded unit
    const unitFilePath = await systemdManager.call("LoadUnit", [unitName]);
    return systemdClient.proxy(
        DBUS_PROPERTIES_INTERFACE,
        unitFilePath[0]
    );
};

/**
 * Fetches all D-Bus properties of a loaded systemd unit.
 *
 * @param {string} unitName - The full name of the unit (e.g., "nodered.service").
 * @returns {Promise<Object>} A promise that resolves to an object of unit properties.
 */
// eslint-disable-next-line no-unused-vars
export const getUnitProperties = async (unitName) => {
    const unitProxy = await getPropertiesProxy(unitName);
    // D-Bus call: GetAll
    // in:  string interface - The interface to get all properties from
    // out: array{sv} props - Array of property name/value pairs
    const response = await unitProxy.call("GetAll", [SYSTEMD_UNIT_INTERFACE]);
    const saneResponse = {};
    for (const [key, value] of Object.entries(response[0])) {
        saneResponse[key] = value.v;
    }
    return saneResponse;
};

/**
 * Asynchronously retrieves the value of a specific property for a given unit.
 *
 * @param {string} unitName - The name of the unit for which the property value is being retrieved.
 * @param {string} propertyName - The name of the property to retrieve from the unit.
 * @returns {Promise<*>} A promise that resolves to the value of the specified property for the given unit.
 */
export const getUnitProperty = async (unitName, propertyName) => {
    const unitProxy = await getPropertiesProxy(unitName);
    const response = await unitProxy.call("Get", [SYSTEMD_UNIT_INTERFACE, propertyName]);
    return response[0].v;
};

/**
 * Retrieves all D-Bus properties for a given systemd service unit.
 *
 * This function uses the systemd D-Bus API to load the specified unit,
 * then queries the `org.freedesktop.systemd1.Service` interface for its properties.
 * The result is returned as a simplified object with plain values.
 *
 * @param {string} unitName - The full name of the service unit (e.g., "nodered.service").
 * @returns {Promise<Object>} A promise that resolves to an object containing the service's properties.
 *
 * @example
 * const props = await getServiceProperties("nodered.service");
 * console.log(props.ExecMainPID, props.ActiveState, props.ExecStart);
 */
export const getServiceProperties = async (unitName) => {
    const unitProxy = await getPropertiesProxy(unitName);
    // D-Bus call: GetAll
    // in:  string interface - The interface to get all properties from
    // out: array{sv} props - Array of property name/value pairs
    const response = await unitProxy.call("GetAll", [SYSTEMD_SERVICE_INTERFACE]);
    const saneResponse = {};
    for (const [key, value] of Object.entries(response[0])) {
        saneResponse[key] = value.v;
    }
    return saneResponse;
};

/**
 * Retrieves a specified property of a systemd service.
 *
 * This asynchronous function fetches the value of a property associated with a systemd service
 * unit. The function uses the unit name and property name to retrieve the desired information.
 *
 * @param {string} unitName - The name of the systemd service unit.
 * @param {string} propertyName - The name of the property to retrieve.
 * @returns {Promise<any>} A promise that resolves to the value of the requested property.
 * @throws {Error} If the property cannot be retrieved or the operation fails.
 */
export const getServiceProperty = async (unitName, propertyName) => {
    const unitProxy = await getPropertiesProxy(unitName);
    const response = await unitProxy.call("Get", [SYSTEMD_SERVICE_INTERFACE, propertyName]);
    return response[0].v;
};

/**
 * Subscribes to changes in the service system for a given unit name.
 * The provided callback function is invoked whenever there is a change.
 *
 * @param {string} unitName - The name of the service unit to monitor for changes.
 * @param {function} onChange - A callback function that is triggered when changes occur.
 * @return {Promise<any>} A Promise that resolves when the subscription is created successfully.
 */
export async function subscribeToServiceChanges (unitName, onChange) {
    subscribe(unitName, SYSTEMD_SERVICE_INTERFACE, onChange);
}

/**
 * Subscribes to changes in the specified unit.
 *
 * @param {string} unitName - The name of the unit to subscribe to.
 * @param {function} onChange - The callback function to invoke when the unit changes.
 * @return {Promise<void>} A promise that resolves once the subscription is successfully created.
 */
export async function subscribeToUnitChanges (unitName, onChange) {
    subscribe(unitName, SYSTEMD_UNIT_INTERFACE, onChange);
}

/**
 * Subscribes to changes on a specific D-Bus unit and invokes a callback
 * function whenever the unit's properties change.
 *
 * @param {string} unitName - The name of the systemd unit to subscribe to.
 * @param {string} dbusInterface - The D-Bus interface used for the subscription.
 * @param {function} onChange - The callback function to handle property change events.
 * It receives the details of the change event as an argument.
 * @return {Promise<void>} A promise that resolves once the subscription has been successfully set up.
 */
async function subscribe (unitName, dbusInterface, onChange) {
    // D-Bus call: LoadUnit
    // in: string name             - The name of the unit to load
    // out: object unit           - Object path of the loaded unit (string)
    const [unitPath] = await systemdManager.call("LoadUnit", [unitName]);
    const unitProxy = systemdClient.proxy(dbusInterface, unitPath);

    // Wait until the proxy is valid (properties loaded)
    await unitProxy.wait();
    unitProxy.addEventListener("changed", (event) => {
        onChange(event.detail);
    });
}

/**
 * Reloads the systemd manager configuration.
 *
 * This is equivalent to running `systemctl daemon-reexec` or `systemctl daemon-reload`,
 * and should be used after modifying unit files to ensure changes are picked up.
 *
 * @returns {Promise<void>} A promise that resolves when the reload operation completes.
 */
export const reloadSystemd = () => {
    systemdManager.call("Reload");
};
