// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import cockpit from "cockpit";
import {
    Alert,
    Button,
    Card,
    CardBody,
    CardFooter,
    CardHeader,
    CardTitle,
    ClipboardCopy, ExpandableSection,
    Flex,
    FlexItem,
    Grid,
    GridItem,
    Modal,
    Switch,
    Text,
    TextContent,
    TextVariants,
    Title,
    Tooltip
} from "@patternfly/react-core";
import { CogIcon, ExternalLinkAltIcon, UserIcon } from "@patternfly/react-icons";
import { disableService, enableService, isServiceActive, isServiceMasked } from "../common/systemd-tools.js";
import { getHostName, readFromFile } from "../common/helper.js";
import {
    activateSandbox,
    addNoderedToGroups, deactivateSandbox,
    removeNoderedFromGroups,
    setLocalDbValue,
    SettingTypes,
    TEMPLATE_SETTINGS_PATH
} from "./revpi-nodered.settings.js";
import {
    AdvancedSetting,
    FlexCenterEnd,
    NODE_RED_STATES,
    REVPI_NODES_STATES,
    useRevpiNodesState
} from "./revpi-nodered.components.jsx";
import { UserManagement } from "./user-management.jsx";
import { useCustomization } from "../branding.js";
import { UiCardTitle, UiDivider, HelpText, SwitchWithSpinner } from "../common/ui-components.jsx";
import { useNotifications } from "../common/hooks.jsx";

const _ = cockpit.gettext;

/**
 * Constants representing the systemd service names
 * for the main Node-RED service and the supporting RevPi nodes server (runtime).
 */
export const NODE_RED_SERVICE = "nodered";
export const NODES_SERVER_SERVICE = "noderedrevpinodes-server";

/**
 * ServiceCard is a memoized React component that renders a card interface
 * for managing the configuration and state of the Node-RED service and its
 * associated components. It provides controls to enable/disable services and
 * displays associated statuses.
 *
 * Props:
 * @param {boolean} isConfigured - Indicates if the application is fully configured
 *        and ready for service management operations.
 * @param {string} nodeRedState - The current state of the Node-RED service,
 *        which determines its operational status (e.g., UP or DOWN).
 *
 * Features:
 * - Displays service status (e.g., Node-RED state).
 * - Allows toggling the Node-RED service on/off.
 * - Automatically starts the Node-RED RevPi Nodes Runtime server if required.
 * - Prevents switching actions when services are masked or unavailable.
 * - Displays Node-RED dashboard URL and supports copying it to the clipboard.
 * - Provides a button to open the Node-RED web interface in a new browser tab.
 */
export const ServiceCard = React.memo(({
    isConfigured,
    nodeRedState,
    canUseSerialInterfaces,
    setCanUseSerialInterfaces,
    canUseAudioVideo,
    setCanUseAudioVideo,
    protectSystem,
    setProtectSystem,
    protectSystemLoading,
    setProtectSystemLoading
}) => {
    const [nodeRedUrl, setNodeRedUrl] = React.useState("");
    const [nodeRedEnabled, setNodeRedEnabled] = React.useState(true);
    const { nodesState } = useRevpiNodesState();
    const [nodeRedMasked, setNodeRedMasked] = React.useState(false);
    const [isAdvancedExpanded, setIsAdvancedExpanded] = React.useState(false);
    const { addNotification } = useNotifications();

    // Fetch service states and dashboard URL on first render
    React.useEffect(() => { // initial page setup
        const fetchServiceState = async (service, setMasked = () => {}) => {
            try {
                const masked = await isServiceMasked(service);
                isServiceActive(service);
                setMasked(masked);
            } catch (error) {
                console.error(`Failed to get ${service} status:`, error);
            }
        };

        const fetchHostName = async () => {
            const createUrl = (hostName) => `https://${hostName}:41880`;
            try {
                while (!cockpit.transport || !cockpit.transport.host) {
                    await new Promise((resolve) => setTimeout(resolve, 100)); // Wait and retry
                }
                setNodeRedUrl(createUrl(getHostName()));
            } catch (error) {
                console.error("Failed to retrieve hostname:", error);
            }
        };

        fetchHostName();
        fetchServiceState(NODE_RED_SERVICE, setNodeRedMasked);
        fetchServiceState(NODES_SERVER_SERVICE);
    }, []);

    React.useEffect(() => {
        setNodeRedEnabled(nodeRedState !== NODE_RED_STATES.DOWN);
    }, [nodeRedState]);

    React.useEffect(() => {
        const _isAdvancedExpanded = window.localStorage.getItem("revpi-nodered:is-advanced-expanded") === "true";
        setIsAdvancedExpanded(_isAdvancedExpanded);
    }, []);

    const handleAdvancedExpandedClick = () => {
        window.localStorage.setItem("revpi-nodered:is-advanced-expanded", !isAdvancedExpanded);
        setIsAdvancedExpanded(!isAdvancedExpanded);
    };

    /**
     * Toggles Node-RED service on or off.
     * If enabling and the secondary server is not running, starts it too.
     */
    const handleNodeRedClick = async () => {
        const isEnabling = nodeRedState === NODE_RED_STATES.DOWN;
        try {
            if (isEnabling) {
                await enableService(NODE_RED_SERVICE);
            } else {
                await disableService(NODE_RED_SERVICE);
            }
            if (isEnabling && nodesState === REVPI_NODES_STATES.DOWN) {
                await handleNodesServerClick();
                addNotification(
                    _("RevPi Nodes Runtime enabled automatically"),
                    _("The RevPi Nodes Runtime feature was enabled automatically because it is required for using RevPi nodes in Node-RED."),
                    "info"
                );
            }
        } catch (error) {
            console.log(error);
        }
    };

    /**
     * Toggles the Node-RED RevPi Nodes Runtime on or off.
     */
    const handleNodesServerClick = async () => {
        const isEnabling = nodesState === REVPI_NODES_STATES.DOWN;
        try {
            if (isEnabling) {
                await enableService(NODES_SERVER_SERVICE);
            } else {
                await disableService(NODES_SERVER_SERVICE);
            }
        } catch (error) {
            console.log(error);
        }
    };

    const handleToggleAudioVideo = async () => {
        const groups = ["audio", "video", "render"];
        try {
            canUseAudioVideo
                ? await removeNoderedFromGroups(groups)
                : await addNoderedToGroups(groups);
            setCanUseAudioVideo(!canUseAudioVideo);
        } catch (error) {
            console.error(error);
        }
    };

    const handleToggleSerialInterfaces = async () => {
        const groups = ["dialout"];
        try {
            canUseSerialInterfaces
                ? await removeNoderedFromGroups(groups)
                : await addNoderedToGroups(groups);
            setCanUseSerialInterfaces(!canUseSerialInterfaces);
        } catch (error) {
            console.error(error);
        }
    };

    const handleToggleProtectSystem = async () => {
        setProtectSystemLoading(true);
        try {
            if (protectSystem) {
                await deactivateSandbox();
                setProtectSystem(false);
            } else {
                await activateSandbox();
                setProtectSystem(true);
            }
        } catch (error) {
            console.error(error);
        } finally {
            setProtectSystemLoading(false);
        }
    };

    function openNodeRed () {
        window.open(nodeRedUrl, "_blank");
    }

    const nodeRedSwitch = (() => {
        const switchElement = nodeRedMasked
            ? (
                <Tooltip
                    content={_("The Node-RED service is masked and cannot be started. Either contact your system administrator or unmask the service yourself using 'systemctl unmask nodered'.")}
                >
                    <Switch
                        label=''
                        isChecked={nodeRedEnabled}
                        isDisabled={!isConfigured}
                        onChange={handleNodeRedClick}
                        isReversed
                    />
                </Tooltip>
            )
            : (
                <Switch
                    label=''
                    isChecked={nodeRedEnabled}
                    isDisabled={!isConfigured}
                    onChange={handleNodeRedClick}
                    isReversed
                />
            );

        return isConfigured
            ? switchElement
            : (
                <Tooltip
                    content={
                        <div>
                            {_("Apply valid settings before starting Node-RED.")}
                        </div>
                    }
                >
                    {switchElement}
                </Tooltip>
            );
    })();

    return (
        <Card>
            <CardHeader>
                <CardTitle>
                    <Flex justifyContent={{ default: "justifyContentSpaceBetween" }}>
                        <FlexItem>
                            <Flex direction={{ default: "row" }}>
                                <FlexItem>
                                    <UiCardTitle>
                                        <span className='node-red-status'>
                                            <span className={`status-dot ${nodeRedState || "down"}`} />
                                            Node-RED
                                        </span>
                                    </UiCardTitle>
                                </FlexItem>
                                <FlexItem>
                                    <HelpText
                                        noSpacing
                                    >
                                        <TextContent>
                                            <Text component='p'>
                                                {_("A simple flow-based tool for connecting devices and automating workflows.")}
                                            </Text>
                                        </TextContent>
                                    </HelpText>

                                </FlexItem>
                            </Flex>

                        </FlexItem>
                        <FlexItem>
                            {nodeRedSwitch}
                        </FlexItem>
                    </Flex>
                    <UiDivider />
                </CardTitle>
            </CardHeader>

            <CardBody style={{
                display: "flex",
                flexDirection: "column"
            }}
            >
                <Grid hasGutter>
                    <GridItem span={6}>
                        <TextContent>
                            <Flex>
                                <FlexItem>{_("App URL")}</FlexItem>
                                <FlexItem grow={{ default: "grow" }}>
                                    <ClipboardCopy
                                        isReadOnly
                                        hoverTip={_("Copy to clipboard")}
                                        clickTip={_("Successfully copied to clipboard!")}
                                    >
                                        {nodeRedUrl}
                                    </ClipboardCopy>
                                </FlexItem>
                            </Flex>

                        </TextContent>
                    </GridItem>
                    <GridItem span={6}>
                        <FlexCenterEnd>
                            <Button
                                variant='primary'
                                icon={<ExternalLinkAltIcon />}
                                iconPosition='right'
                                onClick={() => openNodeRed()}
                                isDisabled={nodeRedState !== NODE_RED_STATES.UP}
                            >
                                {_("Open Node-RED")}
                            </Button>
                        </FlexCenterEnd>

                    </GridItem>
                    <GridItem span={12}>
                        <SwitchWithSpinner
                            label={_("RevPi Nodes Runtime")}
                            isChecked={nodesState === REVPI_NODES_STATES.UP}
                            isLoading={nodesState === REVPI_NODES_STATES.LOADING}
                            isDisabled={nodesState === REVPI_NODES_STATES.LOADING}
                            onChange={handleNodesServerClick}
                            isReversed
                            helpText={
                                <TextContent>
                                    <Text component='p'>
                                        {_("Includes RevPi nodes runtime for Node-RED.")}
                                    </Text>
                                </TextContent>
                            }
                        />
                    </GridItem>
                    {/* Advanced section */}
                    <ExpandableSection
                        style={{ marginTop: "0.5rem" }}
                        toggleText={_("Advanced Settings")}
                        isExpanded={isAdvancedExpanded}
                        onToggle={handleAdvancedExpandedClick}
                    >
                        <AdvancedSetting
                            label={_("Serial interface permissions")}
                            helpText={_("Enables permissions for external serial interfaces (e.g., RS485 for Modbus RTU) in Node-RED. This will restart Node-RED.")}
                            handler={handleToggleSerialInterfaces}
                            isChecked={canUseSerialInterfaces}
                        />
                        <AdvancedSetting
                            label={_("Audio and video permissions")}
                            helpText={_("Enables permissions to use audio and video devices on the RevPi in Node-RED. This will restart Node-RED.")}
                            handler={handleToggleAudioVideo}
                            isChecked={canUseAudioVideo}
                        />
                        <AdvancedSetting
                            label={_("Extended system protection")}
                            helpText={
                                <TextContent>
                                    <Text component='p'>
                                        {_("Extended system protection restricts Node-RED access to essential areas.")}
                                    </Text>
                                    <Text component='p'>
                                        {_("System files are write-protected, allowing changes only in /var/lib/revpi-nodered and /tmp. Sensitive directories such as /boot, /home, and /root are completely blocked, and kernel components are protected from modification.")}
                                    </Text>
                                </TextContent>
                            }
                            handler={handleToggleProtectSystem}
                            isLoading={protectSystemLoading}
                            isChecked={protectSystem}
                        >
                            {!protectSystem && (
                                <Alert
                                    isInline
                                    variant='danger'
                                    title={_("Security Risk")}
                                >{_("Disabling system protection may pose security risks and is not covered by KUNBUS Support.")}
                                </Alert>
                            )}
                        </AdvancedSetting>
                    </ExpandableSection>
                </Grid>

            </CardBody>
        </Card>
    );
});

/**
 * SettingsCard is a React memoized component that provides a user interface for managing various application settings.
 * It includes functionalities to modify settings, confirm changes, and apply those changes. The component uses
 * multiple states and handlers to manage interaction and validation logic.
 *
 * @param {Object} props - The properties passed to the component.
 * @param {Object} props.codeEditor - The reference to the code editor instance used to modify settings.
 * @param {Function} props.setEditorCode - A function to update the code content displayed within the editor.
 * @param {boolean} props.isLocalDbReady - Indicates whether the local database is ready for interactions.
 * @param {Function} props.updateSettingsChanges - A function to synchronize and save updates to settings.
 * @param {boolean} props.settingsChanged - A flag indicating whether the settings have been modified by the user.
 * @param {Array} props.users - The array of users managed by the settings.
 * @param {Function} props.setUsers - A function to set or modify the list of users.
 * @param {boolean} props.useAuthentication - A flag that determines if the authentication feature is enabled.
 * @param {Function} props.setUseAuthentication - A function to toggle the authentication feature.
 * @param {boolean} props.useAdvancedMode - Indicates whether advanced mode options are enabled in the settings.
 * @param {Function} props.setUseAdvancedMode - A function to enable or disable advanced mode options.
 * @param {boolean} props.isApplying - A flag indicating if the settings are currently being applied.
 * @param {boolean} props.canApply - A flag indicating whether the settings can be applied based on validation logic.
 * @param {Function} props.setCanApply - A function to update the state of whether settings can be applied.
 * @param {Function} props.simpleSettingsValid - A function that validates the basic settings to check their readiness.
 * @param {Function} props.handleApplySettings - A function invoked to apply the current settings.
 */
export const SettingsCard = React.memo(
    ({
        codeEditor,
        setEditorCode,
        isLocalDbReady,
        updateSettingsChanges,
        settingsChanged,
        users,
        setUsers,
        useAuthentication,
        setUseAuthentication,
        useAdvancedMode,
        setUseAdvancedMode,
        isApplying,
        canApply,
        setCanApply,
        simpleSettingsValid,
        handleApplySettings
    }) => {
        const [isLoadDefaultsModalOpen, setIsLoadDefaultsModalOpen] = React.useState(false); // used to confirm before loading defaults into editor

        /**
         * Sets the state of the `canApply */
        React.useEffect(() => {
            setCanApply(useAdvancedMode || simpleSettingsValid());
        }, [useAuthentication, users, useAdvancedMode]);

        /**
     * Loads default Node-RED settings from template file
     */
        const handleLoadNodeRedDefaults = async () => {
            const content = await readFromFile(TEMPLATE_SETTINGS_PATH);
            setEditorCode(content);
        };

        /**
     * Toggles between simple and advanced settings mode
     */
        const handleToggleAdvancedMode = () => {
            setLocalDbValue("savedSettingsMode", useAdvancedMode ? SettingTypes.SIMPLE : SettingTypes.ADVANCED);
            setUseAdvancedMode(!useAdvancedMode);
            updateSettingsChanges();
        };

        return (
            <>

                <Card
                    isRounded
                    style={{
                        opacity: isApplying ? 0.7 : 1,
                        pointerEvents: isApplying ? "none" : "auto"
                    }}
                >
                    <CardHeader>
                        <CardTitle>
                            <Flex justifyContent={{ default: "justifyContentSpaceBetween" }}>
                                <FlexItem>
                                    <Flex direction={{ default: "row" }}>
                                        <FlexItem>
                                            <UiCardTitle>
                                                <CogIcon width='36' height='36' style={{ marginRight: "1rem" }} />{_("Node-RED Settings")}
                                            </UiCardTitle>
                                        </FlexItem>
                                    </Flex>

                                </FlexItem>
                                <FlexItem>
                                    <HelpText>
                                        <TextContent>
                                            <Text component='p'>
                                                {_("Expert Mode enables direct editing of the Node-RED settings.js configuration file using the integrated code editor.")}
                                            </Text>
                                        </TextContent>
                                    </HelpText>
                                    <Switch
                                        onChange={handleToggleAdvancedMode}
                                        isChecked={!!useAdvancedMode}
                                        label={_("Expert Mode")}
                                        isReversed
                                    />
                                </FlexItem>
                            </Flex>
                            <UiDivider />
                        </CardTitle>
                    </CardHeader>

                    <CardBody>
                        {
                            useAdvancedMode
                                ? (
                                    <>
                                        <Alert
                                            isInline
                                            variant='warning'
                                            title={
                                                <Flex alignItems={{ default: "alignItemsCenter" }}>
                                                    <FlexItem>
                                                        {_("Changes are made at your own risk and are not covered by KUNBUS Support.")}
                                                    </FlexItem>
                                                    <FlexItem>
                                                        <HelpText
                                                            noSpacing
                                                        >
                                                            <TextContent>
                                                                <Text component='p'>
                                                                    {_("Direct editing of settings.js through the integrated code editor, intended for experienced users.")}
                                                                </Text>
                                                                <Text component='p'>
                                                                    {_("Incorrect changes to the settings.js file may prevent Node-RED from starting properly.")}
                                                                </Text>
                                                                <Text component='p'>
                                                                    {_("Use the ")}
                                                                    <i>{_("Load Node-RED defaults")}</i>
                                                                    {_(" button to restore the default version of the settings.js file if needed.")}
                                                                </Text>
                                                                <Text component='p'>
                                                                    {_("Changes made in the code editor are preserved when leaving Expert Mode.")}
                                                                </Text>
                                                                <Text component='p'>
                                                                    <strong>{_("Do not modify the options 'uiHost', 'uiPort', 'https' and 'requireHttps' as they are managed externally.")}</strong>
                                                                </Text>
                                                            </TextContent>
                                                        </HelpText>
                                                    </FlexItem>
                                                </Flex>
                                            }
                                        />
                                        {codeEditor}
                                        {/* <CodeEditor code={code} setCode={setCode} updateSettingsChanges={updateSettingsChanges} /> */}
                                    </>
                                )
                                : (
                                    <SimpleSettings
                                        isLocalDbReady={isLocalDbReady}
                                        updateSettingsChanges={updateSettingsChanges}
                                        useAuthentication={useAuthentication}
                                        setUseAuthentication={setUseAuthentication}
                                        users={users}
                                        setUsers={setUsers}
                                    />
                                )
                        }
                    </CardBody>

                    <CardFooter>
                        <Flex
                            spaceItems={{ default: "spaceItemsMd" }}
                            direction={{ default: "column" }}
                        >
                            <FlexItem>
                                <Alert
                                    isInline
                                    variant={canApply ? "info" : "warning"}
                                    style={{ visibility: (settingsChanged || !canApply) ? "visible" : "hidden" }}
                                    title={canApply ? _("Settings not applied.") : _("Add a user or disable authentication.")}
                                />
                            </FlexItem>

                            <FlexItem
                                alignSelf={{ default: "alignSelfFlexEnd" }}
                            >
                                {
                                    useAdvancedMode &&
                                    <Button
                                        style={{ marginRight: "1rem" }}
                                        variant='secondary'
                                        onClick={() => setIsLoadDefaultsModalOpen(true)}
                                    >
                                        {_("Load Node-RED defaults")}
                                    </Button>
                                }

                                <Button
                                    isDisabled={isApplying || !canApply} id='apply_button' variant='primary'
                                    onClick={handleApplySettings}
                                >
                                    {_("Apply Settings")}
                                </Button>
                            </FlexItem>
                        </Flex>
                    </CardFooter>
                </Card>
                <Modal
                    title={_("Restore Default Settings")}
                    isOpen={isLoadDefaultsModalOpen}
                    onClose={() => setIsLoadDefaultsModalOpen(false)}
                    variant='small'
                    actions={[
                        <Button
                            key='cancel'
                            variant='secondary'
                            onClick={() => setIsLoadDefaultsModalOpen(false)}
                        >
                            {_("Cancel")}
                        </Button>,
                        <Button
                            key='confirm'
                            variant='danger'
                            onClick={() => {
                                handleLoadNodeRedDefaults();
                                setIsLoadDefaultsModalOpen(false);
                            }}
                        >
                            {_("Restore")}
                        </Button>
                    ]}
                >
                    <TextContent>
                        <Text component='p'>
                            {_("This will overwrite any changes made in Expert Mode and restore the default settings.js file.")}
                        </Text>
                        <Text component='p'>
                            <strong>
                                {_("This action cannot be undone. Are you sure you want to continue?")}
                            </strong>
                        </Text>
                    </TextContent>
                </Modal>
            </>
        );
    });

export const SimpleSettings = ({
    useAuthentication,
    setUseAuthentication,
    updateSettingsChanges,
    isLocalDbReady,
    users,
    setUsers
}) => {
    const [isAddUserModalOpen, setIsAddUserModalOpen] = React.useState(false);
    const openModal = () => setIsAddUserModalOpen(true);
    const closeModal = () => setIsAddUserModalOpen(false);
    const { productName } = useCustomization();
    /**
     * Toggles authentication setting and updates settings state to prompt for save
     */
    const handleUseAuthentication = async () => {
        setLocalDbValue("useAuthentication", !useAuthentication);
        await updateSettingsChanges();
        setUseAuthentication(!useAuthentication);
    };

    return (
        <Grid hasGutter>
            <GridItem span={12}>
                <Flex direction={{ default: "row" }}>
                    <FlexItem>
                        <SwitchWithSpinner
                            isChecked={useAuthentication}
                            isLoading={!isLocalDbReady}
                            onChange={handleUseAuthentication}
                            label={_("Require Authentication")}
                            isReversed
                            helpText={
                                <TextContent>
                                    <Text component='p'>
                                        {cockpit.format(_(
                                            "Enable this option for security reasons. Without it, anyone with network access to your $0 device can access the Node-RED editor."
                                        ), productName)}
                                    </Text>
                                </TextContent>
                            }
                        />
                    </FlexItem>
                    <FlexItem grow={{ default: "grow" }}>
                        <Alert
                            style={{
                                visibility: useAuthentication ? "hidden" : "visible"
                            }}
                            isInline
                            variant='danger'
                            title={_("Security Risk")}
                        >{_("With authentication disabled, the Node-RED editor is openly accessible to anyone on the network.")}
                        </Alert>
                    </FlexItem>
                </Flex>
            </GridItem>

            <GridItem span={8}>
                <TextContent>
                    <Title headingLevel='h2' size='lg'>{_("User Management")}</Title>
                    <Text component={TextVariants.small}>
                        {_("Manage which users have access to Node-RED and their permissions.")}
                    </Text>
                </TextContent>
            </GridItem>

            <GridItem span={4}>
                <Flex justifyContent={{ default: "justifyContentFlexEnd" }} alignItems={{ default: "alignItemsFlexEnd" }} style={{ height: "100%" }}>
                    <Button
                        variant='primary'
                        onClick={openModal}
                        style={{ marginRight: "1rem" }}
                        icon={<UserIcon />}
                    >
                        {_("Add User")}
                    </Button>
                </Flex>
            </GridItem>

            <GridItem>
                <UserManagement
                    users={users}
                    setUsers={setUsers}
                    updateSettingsChanges={updateSettingsChanges}
                    useAuthentication={useAuthentication}
                    isModalOpen={isAddUserModalOpen}
                    closeModal={closeModal}
                />
            </GridItem>
        </Grid>
    );
};
