<!--
SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH

SPDX-License-Identifier: GPL-2.0-or-later
-->

# revpi-config Cockpit Module

This module provides RevPi specific configuration options and access to the pictory web app.

## Pictory SSO Login
It is possible to seamlessly log into pictory 
from the revpi-config module without the need for any other authentication.

## SSO Authentication Details
To authenticate a user in pictory based on an active cockpit session is done via token exchange on 
the filesystem.
A cockpit session runs in the users context on the device and can create
a token file in the users name. Those tokens are created in the folder `/run/cockpit-revpi-pictory-sso` and consist of
a UUID as filename and the username as content.
Cockpit will pass this UUID with the link to pictory's sso_login Route which checks if
a token for this UUID exists and if so, creates an authenticated session for the user.
The token is deleted afterwards so it can only be claimed once.

## The /run/cockpit-revpi-pictory-sso directory
This directory is created by [systemd-tmpfiles](https://www.freedesktop.org/software/systemd/man/latest/systemd-tmpfiles.html#)
according to the configuration in `src/revpi-config/systemd/cockpit-revpi-pictory-sso.tmpfiles.conf`.
The conf file is copied to `/usr/local/lib/tmpfiles.d` from where systemd picks it up.

*Note:* If you are using the RSYNC option during development, you need to copy this file manually.
After deploying at least once to the target run this command on the target
```shell
$ sudo cp ~/.local/share/cockpit/revpi-config/systemd/cockpit-revpi-pictory-sso.tmpfiles.conf /usr/local/lib/tmpfiles.d/
# Restart systemd-tmpfiles 
$ sudo systemd-tmpfiles --create 
# make sure the folder was created
$ ls -alh /run/cockpit-revpi-pictory-sso
```

## React development guide

1. JSX – The Basics
   JSX (JavaScript XML) is a syntax extension for JavaScript that allows you to 
   write HTML-like code inside your React components. It makes it easier to
   visualize the UI structure.

Example:
```
const MyFunctionalComponent = () => {
    return <h1>Hello, World!</h1>;
};
```
JSX must always return a single parent element, so wrap multiple elements
inside a <div> or React fragment (<>...</>).

2. useState – Managing Component State
   useState is a React hook that allows functional components to manage local state.

Example:

```
import React from "react";

const Counter = () => {
    const [count, setCount] = React.useState(0);

    return (
        <div>
            <p>Count: {count}</p>
            <button onClick={() => setCount(count + 1)}>Increase</button>
        </div>
    );
};

```
* React.useState(0) initializes count to 0.
* setCount(count + 1) updates the state when the button is clicked.

3. useEffect – Handling Side Effects
   useEffect allows you to run code after rendering (e.g., fetching data, setting up 
   subscriptions, updating the document title).

Example:

``` 
import React from "react";

const Timer = () => {
    const [seconds, setSeconds] = React.useState(0);

    React.useEffect(() => {
        const interval = setInterval(() => {
            setSeconds((prev) => prev + 1);
        }, 1000);

        return () => clearInterval(interval); // Cleanup function
    }, []); // Empty array = runs only once after mount

    return <p>Time: {seconds} seconds</p>;
};

```

* The effect runs once when the component mounts ([] dependency array).
* clearInterval(interval) ensures the timer stops when the component unmounts.


4. Optimizing Performance – React.useCallback & React.useMemo
   As your app grows, unnecessary re-renders can slow it down. React.useCallback and React.useMemo help optimize performance.


**useCallback – Memoizing Functions** \
Prevents function recreation on every render.

```
import React from "react";

const Button = ({ onClick }) => <button onClick={onClick}>Click me</button>;

const Parent = () => {
    const [count, setCount] = React.useState(0);

    const handleClick = React.useCallback(() => {
        setCount((prev) => prev + 1);
    }, []); // Function reference stays the same

    return <Button onClick={handleClick} />;
};
```

Without useCallback, a new handleClick function would be created on every render.

**useMemo – Caching Expensive Calculations** \
Avoids recalculating values unnecessarily.

```
import React from "react";

const ExpensiveCalculation = ({ number }) => {
    const result = React.useMemo(() => {
        console.log("Calculating...");
        return number * 2; // Simulated expensive computation
    }, [number]); // Recalculates only when `number` changes

    return <p>Result: {result}</p>;
};

```

### Conclusion
* JSX helps define UI in a readable way.
* useState manages local component state.
* useEffect handles side effects (e.g., API calls, event listeners).
* useCallback and useMemo optimize performance by avoiding unnecessary re-renders.


## RevPi Configuration Plugin Structure

### 1. `app.jsx` – The Main Parent Component
The core logic of the **RevPi Configuration Plugin** is contained in `app.jsx`. It acts as the **top-level UI controller** and manages:
- The **feature state**:
  ```jsx
  const [features, setFeatures] = React.useState(new Map());
  ```
  This stores and updates the listed features from `RevPiConfigData`.
- **Loading and error states**, ensuring that UI feedback is properly handled.

Additionally, `app.jsx` is wrapped inside `ApplicationRoot`, which provides:
- **Toast notifications anywhere in the app**
- **The default Cockpit page styling**

```jsx
export const ApplicationRoot = ({ children }) => (
    <NotificationProvider>
        <Page>
            <PageSection variant={PageSectionVariants.default}>
                <Notifications />
                {children}
            </PageSection>
        </Page>
    </NotificationProvider>
);
```

This ensures that any component inside `ApplicationRoot` can trigger notifications globally.

### 2. Managing UI and Feature Components
`app.jsx` is responsible for rendering **configuration cards** that represent different setting categories:

- `PictoryCard`
- `NodeRedCard`
- `SystemConfigurationCard`
- `NetworkCard`
- `RunTimeCard`

Each card receives the **necessary feature information** from `app.jsx`, ensuring consistent state updates. \
Features like cards are located in the 'features' directory.

### 3. Common Components in `common.revpi-nodered.common.components.jsx`
Reusable UI components that are used throughout the application are located in:

```
common.revpi-nodered.common.components.jsx
```

One example is **`RevPiSwitch`**, a **toggle element** that allows users to change `revpi-config` states.

Example:
```jsx
export const RevPiSwitch = (props) => {
    const { id, isEnabled, isLoading } = props.item;

    return (
        // RevPiFlexBase is handling the layout
        <RevPiFlexBase {...props}>
            // Switch component from patternfly (UI library)
            <Switch
                hasCheckIcon
                id={id}
                label={_("Enabled")}
                labelOff={_("Disabled")}
                isChecked={isEnabled}
                isDisabled={isLoading}
                onChange={() => {
                    props.switchHandler(id, isEnabled, isLoading);
                }}
            />
        </RevPiFlexBase>
    );
};
```

By centralizing commonly used components, this structure ensures **reusability and maintainability**.

### 4. System Operations with `revpi-config-service.jsx`
Logic that interacts with the **system** using Cockpit’s interfaces, such as:
- Running **commands with `cockpit.spawn`** like **`revpi-config` scripts** for system configuration

is located in:

```
revpi-config/revpi-config-service.jsx
```

Example of calling a `revpi-config` script:
```jsx
const callRevpiConfig = async (command, ids = []) => {
    return cockpit.spawn(["/usr/bin/revpi-config", command, ...ids], { superuser: "require" });
};
```
This ensures **clean separation between UI logic and system operations**, improving maintainability.

### 5. Global Event Communication with `emitters.js`
Some interactions in the application require **cross-component communication** where passing props is not sufficient.  
For such cases, **Cockpit's event emitters** are used via:

```
emitters.js
```

Example:
```jsx
import cockpit from "cockpit";

export const installEventEmitter = cockpit.event_target({
    events: {
        start: "installation-started",
        finish: "installation-finished"
    }
});
```

This allows components to:
- **Listen for installation events** without requiring direct prop drilling.
- **Trigger refreshes** in `app.jsx` after an installation completes.

Example usage in `app.jsx`:
```jsx
useEffect(() => {
    installEventEmitter.addEventListener(
        installEventEmitter.events.finish,
        loadAllFeatures
    );

    return () => {
        installEventEmitter.removeEventListener(
            installEventEmitter.events.finish,
            loadAllFeatures
        );
    };
}, [loadAllFeatures]);
```
This ensures that `loadAllFeatures` is **triggered globally** whenever an installation finishes.

### 6. Handling Permissions and Feature Display
Inside the `useEffect` section:
- **User permissions are checked** using:
  ```jsx
  const permission = cockpit.permission({ admin: true });
  ```
- **Features are conditionally displayed** based on the user's permission level.
- If access is denied, an error message is shown.
