// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import {
    Alert,
    AlertActionCloseButton,
    AlertGroup,
    Button,
    Card,
    CardBody,
    CardHeader,
    CodeBlock,
    CodeBlockCode,
    Divider,
    Flex,
    FlexItem,
    FormGroup,
    MenuToggle,
    Popover,
    Select,
    SelectList,
    SelectOption,
    Spinner,
    Stack,
    Switch,
    TextInput,
    Title
} from "@patternfly/react-core";
import React from "react";
import { ExclamationCircleIcon, OutlinedQuestionCircleIcon, SpinnerIcon } from "@patternfly/react-icons";

import cockpit from "cockpit";
import { useFeatureInstaller, useNotifications } from "./hooks.jsx";
import { HelperText, HelperTextItem } from "@patternfly/react-core/dist/esm/components/HelperText/index.js";
import { EmptyStatePanel } from "../../pkg/lib/cockpit-components-empty-state.jsx";

const _ = cockpit.gettext;

/**
 * Display types for feature buttons in the UI.
 * Note: Translations are applied to dynamic string values, which requires
 * having translations in poupdate. Direct translation calls in this object
 * are not functional.
 */
export const DISPLAY_TYPES = {
    OPEN: _("Open"),
    CONFIGURE: _("Configure"),
    INSTALL: _("Install")
};
export const DISPLAY_TYPE_COLORS = {
    [DISPLAY_TYPES.OPEN]: "primary",
    [DISPLAY_TYPES.CONFIGURE]: "tertiary",
    [DISPLAY_TYPES.INSTALL]: "secondary"
};

/**
 * A functional React component that renders a `Divider` element with a
 * predefined custom style.
 *
 * The purpose of `UiDivider` is to provide a reusable divider component
 * that includes a specific margin applied to the top of the element.
 *
 * Styling:
 * - `marginTop`: 0.75rem
 *
 * Returns:
 * - A React element representing a styled `Divider`.
 */
export const UiDivider = ({ style, ...props }) => (
    <Divider style={{ marginTop: "0.75rem", ...style }} {...props} />
);

export const UiCardTitle = ({ children }) => {
    return (
        <Title headingLevel='h1' size='2xl'>
            {children}
        </Title>
    );
};

/**
 * Renders a help text popover with customizable title, content, footer, and spacing options.
 *
 * @param {Object} props - The properties object.
 * @param {React.ReactNode} props.title - The title content displayed in the popover header.
 * @param {React.ReactNode} props.children - The main body content of the popover.
 * @param {React.ReactNode} props.footer - The footer content displayed in the popover.
 * @param {boolean} [props.noSpacing=false] - Optional flag to remove padding and margin from the button.
 *
 * @return {JSX.Element} A React component rendering a button that opens a popover.
 */
export function HelpText ({
    title,
    children,
    footer,
    noSpacing = false
}) {
    return (
        <Popover
            headerContent={<div>{title}</div>}
            bodyContent={<div>{children}</div>}
            footerContent={<div>{footer}</div>}
        >
            <Button
                style={noSpacing
                    ? {
                        padding: 0,
                        margin: 0
                    }
                    : undefined}
                variant='link'
                icon={<OutlinedQuestionCircleIcon />}
            />
        </Popover>
    );
}

/**
 * Renders a feature installation card. The card displays a message and an install
 * button if the specified feature is not installed. If the feature is already installed,
 * the children elements are rendered instead.
 *
 * @param {Object} params The parameters for rendering the installation card.
 * @param {string} params.featureName The name of the feature to be checked and installed.
 * @param {string[]} params.packagesName The name of the packages associated with the feature.
 * @param {React.ReactNode} params.children The child components to display when the feature is installed.
 * @return {React.Element} The rendered installation card or children if the feature is installed.
 */
export function InstallFeatureCard ({
    featureName,
    packagesName,
    children
}) {
    const {
        isInstalled,
        isInstalling,
        handleInstall
    } = useFeatureInstaller(featureName, packagesName);

    if (isInstalled) return <>{children}</>;

    return (
        <Card isPlain>
            <CardHeader>
                <Alert
                    variant='info'
                    isInline
                    isPlain
                    title={cockpit.format(_("$0 is not installed."), featureName)}
                />
            </CardHeader>
            <CardBody>
                <Button
                    onClick={handleInstall}
                    isDisabled={isInstalling}
                    variant={DISPLAY_TYPE_COLORS[DISPLAY_TYPES.INSTALL]}
                >
                    {cockpit.format(_("Install $0"), featureName)}
                </Button>
            </CardBody>
        </Card>
    );
}

/**
 * A functional component that combines a switch and a spinner, with optional help text.
 * This component provides a toggle switch accompanied by a spinner to indicate a loading state or help text for additional information.
 *
 * @param {Object} props - Properties passed to the component.
 * @param {string} props.label - Label displayed when the switch is on.
 * @param {string} props.labelOff - Label displayed when the switch is off.
 * @param {boolean} props.isDisabled - Indicates whether the switch is disabled.
 * @param {boolean} props.isChecked - Indicates whether the switch is in the checked state.
 * @param {Function} props.onChange - Callback function invoked when the switch is toggled.
 * @param {string} props.id - ID of the switch.
 * @param {boolean} props.isReversed - If true, reverses the order of the label and switch.
 * @param {boolean} props.isLoading - If true, displays the spinner next to the switch.
 * @param {string} [props.helpTextTitle] - Title for the help text displayed when not loading.
 * @param {string} [props.helpText] - Additional help text displayed when not loading.
 * @returns {JSX.Element} A combination of a switch and a spinner, or help text.
 */
export const SwitchWithSpinner = (props) => (
    <Flex alignItems={{ default: "alignItemsCenter" }}>
        <FlexItem>
            <Switch
                className={props.isAligned ? "spinner-min-width" : undefined}
                label={props.label}
                labelOff={props.labelOff}
                isDisabled={props.isDisabled}
                isChecked={props.isChecked}
                onChange={() => props.onChange()}
                id={props.id}
                isReversed={props.isReversed}
            />
        </FlexItem>
        <FlexItem>
            <div style={{
                width: "24px",
                height: "24px",
                display: "flex",
                alignItems: "center",
                justifyContent: "center"
            }}
            >
                {props.isLoading
                    ? (
                        <Spinner size='lg' />
                    )
                    : (
                        (props.helpTextTitle || props.helpText) && (
                            <HelpText noSpacing title={props.helpTextTitle}>{props.helpText}</HelpText>
                        )
                    )}
            </div>
        </FlexItem>
    </Flex>
);

/**
 * A React functional component that renders a group of toast-style notifications using the AlertGroup component. Each notification can be individually closed with a close button.
 *
 * @return {JSX.Element} The rendered AlertGroup component containing a list of Alert components with notifications.
 */
export function Notifications () {
    const {
        notifications,
        removeNotification
    } = useNotifications();

    return (
        <AlertGroup isToast>
            {notifications.map(notification => (
                <Alert
                    key={notification.id}
                    variant={notification.variant}
                    title={notification.title}
                    actionClose={<AlertActionCloseButton onClose={() => removeNotification(notification.id)} />}
                >
                    {notification.message}
                </Alert>
            ))}
        </AlertGroup>
    );
}

/**
 * Renders an error icon with accompanying helper text.
 *
 * @return {JSX.Element} A React component that displays an error message with an icon, wrapped in a helper text container.
 */
export function ErrorIcon ({ errorText }) {
    return (
        <HelperText>
            <HelperTextItem variant='error' hasIcon>
                {errorText}
            </HelperTextItem>
        </HelperText>
    );
}

/**
 * A functional React component that displays an error or loading message.
 *
 * This component renders a message based on the provided `loading` and `error` parameters. When `loading` is true, a loading spinner and corresponding message
 * are displayed. When an `error` occurs, an error icon and related message are shown, including additional detailed error text and code, if available.
 *
 * @param {Object} props - The properties passed to the component.
 * @param {boolean} props.loading - Indicates whether the component is in a loading state.
 * @param {string|null} props.error - A string defining the error state. Possible values include specific error identifiers (e.g., "access-denied") or null
 * if no error occurred. If the value is "access-denied," an additional message about required permissions will be displayed.
 *
 * @returns {JSX.Element} A JSX element displaying the error message, loading message, or additional information.
 */
export const ErrorMessage = ({
    loading,
    error,
    children
}) => {
    return (
        <div>
            <EmptyStatePanel
                icon={error ? ExclamationCircleIcon : SpinnerIcon}
                loading={loading}
                title={error ? _("Something went wrong") : _("Loading. Please wait...")}
                paragraph={
                    error && (
                        <Stack hasGutter>
                            <div>{_("There was an unexpected error.")}</div>
                            {error === "access-denied" &&
                            <div>{_("Error: Root permissions required for this page.")}</div>}
                            <CodeBlock
                                style={{
                                    backgroundColor: "var(--pf-v5-global--BackgroundColor--light-100)",
                                    border: "1px solid var(--pf-v5-global--palette--black-300)", // subtle border for contrast
                                    borderRadius: "4px",
                                    padding: "0.5rem"
                                }}
                            >
                                <CodeBlockCode
                                    id='code-content'
                                >{error}
                                </CodeBlockCode>
                            </CodeBlock>
                        </Stack>)
                }
            />
            {children}
        </div>
    );
};

/**
 * A functional React component to render a MAC address input field with formatting,
 * validation, and a locked prefix. This component ensures the MAC address is formatted
 * and validated against the defined groups and separator while enforcing a locked prefix.
 *
 * @param {Object} params - Component parameters.
 * @param {string} params.id - The unique identifier for the input field.
 * @param {string} [params.label="Enter your device's MAC address"] - The label displayed for the input field.
 * @param {boolean} [params.required=true] - Whether the input field is mandatory.
 * @param {string} [params.initial="C83E-A7"] - The initial MAC address value to prefill the input.
 * @param {number[]} [params.groups=[4, 4, 4]] - Array defining the size of address segments; concatenated with the separator.
 * @param {string} [params.separator="-"] - The character used to separate MAC address segments.
 * @param {function(string): void} [params.onChange] - Callback function triggered on changes to the MAC address input. Receives the normalized address string.
 *
 * @return {JSX.Element} A React JSX element representing the formatted MAC address input field.
 */
export function MacAddressInput ({
    id,
    label = "Enter your device's MAC address",
    required = true,
    initial = "C83E-A7",
    groups = [4, 4, 4], // XXXX-SSSS-TTTT
    separator = "-",
    onChange
}) {
    /* ---------- helpers ---------- */
    const needed = React.useMemo(() => groups.reduce((a, b) => a + b, 0), [groups]);
    const normalizeHex = (raw) => (raw ?? "").replace(/[^0-9a-fA-F]/g, "").toUpperCase();

    const formatWithGroups = (hex) => {
        let i = 0;
        const parts = groups.map((g) => {
            const part = hex.slice(i, i + g);
            i += g;
            return part.toUpperCase();
        });
        const lastFilled = parts.reduceRight((idx, p, i) => (idx === -1 && p.length > 0 ? i : idx), -1);
        return parts.slice(0, lastFilled + 1).join(separator);
    };

    const isValidHex = (hex) => /^[0-9A-F]+$/.test(hex);

    /* ---------- state & locked prefix ---------- */
    const initNorm = normalizeHex(initial);
    const LOCK_LEN = 6;
    const lockedPrefix = initNorm.slice(0, LOCK_LEN);

    // Initialize display from initial (already contains the prefix)
    const [display, setDisplay] = React.useState(formatWithGroups(initNorm));
    const [touched, setTouched] = React.useState(false);

    /* ---------- derive normalized from display ---------- */
    const normalizedRaw = normalizeHex(display); // may miss the prefix if user tried to delete
    // Enforce the locked prefix no matter what is in display:
    const tail = normalizedRaw.startsWith(lockedPrefix)
        ? normalizedRaw.slice(LOCK_LEN)
        : normalizedRaw.replace(new RegExp(`^${lockedPrefix}`), ""); // strip accidental dup prefix
    const normalized = (lockedPrefix + tail).slice(0, needed); // clamp to total length

    const isComplete = normalized.length === needed && isValidHex(normalized);
    const validNow = isComplete || (!required && normalized.length === 0);
    const validated = touched ? (validNow ? "success" : "error") : "default";

    /* ---------- effects ---------- */
    React.useEffect(() => {
        onChange?.(normalized);
    }, [normalized, onChange]);

    const handleChange = (a, b) => {
        const raw = typeof a === "string" ? a : b;
        const hexOnly = normalizeHex(raw);

        // 👇 Always take only what comes AFTER the locked prefix
        const userTail = hexOnly.slice(LOCK_LEN);
        const enforced = (lockedPrefix + userTail).slice(0, needed);

        setDisplay(formatWithGroups(enforced));
    };

    const handleBlur = () => setTouched(true);

    /* ---------- UX strings ---------- */
    const maskFromGroups = groups.map((g) => "X".repeat(g)).join(separator); // "XXXX-SSSS-TTTT"
    const placeholder = maskFromGroups;

    const helperText =
        validated === "error"
            ? `Invalid value. Expected ${needed} hex characters in the format ${maskFromGroups}.`
            : "Type the Mac address you find on your device's front next to A (below the LAN cable). The 6 first digits are prefilled and fixed.";

    /* ---------- render ---------- */
    return (
        <FormGroup label={label} fieldId={id} isRequired={required} validated={validated}>
            <TextInput
                id={id}
                value={display}
                onChange={handleChange}
                onBlur={handleBlur}
                placeholder={placeholder}
                autoCapitalize='characters'
                autoCorrect='off'
                spellCheck={false}
                validated={validated}
                aria-invalid={validated === "error" ? true : undefined}
            />
            <HelperText>
                <HelperTextItem variant={validated === "error" ? "error" : "default"}>
                    {helperText}
                </HelperTextItem>
            </HelperText>
        </FormGroup>
    );
}

/**
 * A React component that renders a form for entering and validating a serial number.
 * Allows numeric input only and enforces a minimum length requirement.
 *
 * @param {Object} props The properties passed to the form component.
 * @param {function(string, boolean):void} props.onChange A callback function invoked when the serial number input is updated.
 *                                                         The first argument is the cleaned serial number, and the second
 *                                                         argument indicates whether the input is valid.
 * @return {JSX.Element} The JSX element representing the serial number form.
 */
export function SerialNumberForm ({ onChange }) {
    const [serialNumber, setSerialNumber] = React.useState("");
    const [isSerialValid, setIsSerialValid] = React.useState(null); // null = untouched

    const handleSerialInput = (_e, value /* value: string */) => {
        // keep digits only
        const cleaned = value.replace(/\D/g, "");
        setSerialNumber(cleaned);

        // valid if input is empty OR consists only of digits
        const valid = /^\d*$/.test(cleaned);

        // show neutral state when empty, error otherwise if invalid
        setIsSerialValid(cleaned.length === 0 ? null : valid);

        if (onChange) {
            onChange(cleaned, valid);
        }
    };

    const validated =
        isSerialValid === null ? "default" : isSerialValid ? "success" : "error";

    return (
        <FormGroup label={_("Serial Number")} isRequired fieldId='serial-number'>
            <TextInput
                id='serial-number'
                value={serialNumber}
                onChange={handleSerialInput}
                inputMode='numeric'
                pattern='^[0-9]+$'
                autoComplete='off'
                spellCheck={false}
                validated={validated}
            />
            {isSerialValid === false && (
                <HelperText>
                    <HelperTextItem variant='error'>
                        {_("Must be numbers only.")}
                    </HelperTextItem>
                </HelperText>
            )}
        </FormGroup>
    );
}

/**
 * A custom dropdown selection component that allows users to select an option from a list.
 * Provides functionality to toggle the dropdown, select an option, and manage its state.
 *
 * @param {Object} params - The parameters for the dropdown component.
 * @param {string} params.value - The currently selected value. Should match a key from `options`.
 * @param {function} params.onChange - A callback function triggered when the selected value changes.
 * @param {Object} params.options - A key-value pair object containing the available options for selection. Keys represent the value while values represent the display labels.
 * @param {string} [params.id="revpi-select"] - The HTML id for the select component.
 * @param {string} [params.placeholder="Select..."] - The placeholder text displayed when no value is selected.
 * @param {boolean} [params.isDisabled=false] - If true, disables interaction with the component.
 *
 * @return {JSX.Element} A dropdown selection component.
 */
export function RevPiSelect ({
    value,
    onChange,
    options,
    id = "revpi-select",
    placeholder = _("Select..."),
    isDisabled = false
}) {
    const [open, setOpen] = React.useState(false);

    const handleToggle = (isOpen) => setOpen(isOpen);
    const handleSelect = (_e, newValue) => {
        onChange?.(newValue);
        setOpen(false);
    };

    const label = value && options[value] ? options[value] : placeholder;

    return (
        <div>
            <Select
                id={id}
                isOpen={open}
                onOpenChange={handleToggle}
                selected={value}
                onSelect={handleSelect}
                toggle={(toggleRef) => (
                    <MenuToggle
                        ref={toggleRef}
                        onClick={() => setOpen((prev) => !prev)}
                        isExpanded={open}
                        isDisabled={isDisabled}
                    >
                        {label}
                    </MenuToggle>
                )}
            >
                <SelectList>
                    {Object.entries(options).map(([key, lbl]) => (
                        <SelectOption key={key} value={key} isSelected={value === key}>
                            {lbl}
                        </SelectOption>
                    ))}
                </SelectList>
            </Select>
        </div>
    );
}
