// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import cockpit from "cockpit";
import {
    Button,
    Grid, GridItem,
    Page,
    PageSection,
    PageSectionVariants
} from "@patternfly/react-core";
import {
    anyHashExists,
    canNoderedUseAudioVideo,
    canNoderedUseSerialInterfaces,
    defaultSettingsJsHash,
    getLocalDbValue,
    getSettingsJsChecksum,
    init,
    isLocalDbInSync,
    isSandbox,
    readSettingsJsContent,
    SettingTypes,
    writeSettings
} from "./revpi-nodered.settings.js";
import { NODE_RED_SERVICE, ServiceCard, SettingsCard } from "./cards.jsx";
import { CodeEditor, FullscreenToggle, NODE_RED_STATES, useNodeRedState } from "./revpi-nodered.components.jsx";
import { restartService } from "../common/systemd-tools.js";

import { AdminAccessButton, NotificationProvider, useAdminPermission } from "../common/hooks.jsx";
import { Notifications } from "../common/ui-components.jsx";
const _ = cockpit.gettext;

export const Application = () => {
    const [isConfigured, setIsConfigured] = React.useState(false); // has this plugin been used to configure Node-RED
    const [isLocalDbReady, setIsLocalDbReady] = React.useState(false); // has data been loaded from the db
    const { nodeRedState } = useNodeRedState(); // state of the Node-RED service (up, down, starting)
    const isAdmin = useAdminPermission(); // is the user an admin?
    const [settingsChanged, setSettingsChanged] = React.useState(false);
    const [useAdvancedMode, setUseAdvancedMode] = React.useState(null);
    const [canUseSerialInterfaces, setCanUseSerialInterfaces] = React.useState(false);
    const [canUseAudioVideo, setCanUseAudioVideo] = React.useState(false);
    const [protectSystem, setProtectSystem] = React.useState(true);
    const [protectSystemLoading, setProtectSystemLoading] = React.useState(false);

    // ui states
    const [isApplying, setIsApplying] = React.useState(false); // used to display a "currently saving"
    const [canApply, setCanApply] = React.useState(false);

    // settings states
    const [useAuthentication, setUseAuthentication] = React.useState(false);
    const [users, setUsers] = React.useState([]);

    /**
     * Compares current in-memory settings with saved ones.
     * If they differ, sets `settingsChanged = true`.
     * This triggers a UI element to notify the user about the local changes.
     */
    const updateSettingsChanges = async () => {
        const changed = await isLocalDbInSync();
        setSettingsChanged(!changed);
    };

    /**
     * Validates the current configuration before applying.
     * Ensures at least one user exists when auth is enabled.
     */
    const simpleSettingsValid = () => {
        if (useAuthentication === true && users.length === 0) {
            return false;
        }
        return true;
    };

    /**
     * Writes settings to disk, restarts Node-RED if needed,
     * and updates the UI to visually display the "applying" state.
     */
    const handleApplySettings = async () => {
        if (!useAdvancedMode) {
            // validate if settings were applied in simple mode
            if (!simpleSettingsValid()) {
                console.log("Simple settings invalid");
                return;
            }
        }
        setIsApplying(true); // Trigger UI loading state while settings are being applied
        const settingsType = useAdvancedMode ? SettingTypes.ADVANCED : SettingTypes.SIMPLE;
        await writeSettings(settingsType);

        if (nodeRedState !== NODE_RED_STATES.DOWN) {
            await restartService(NODE_RED_SERVICE);
        }
        setIsConfigured(anyHashExists()); // check whether the plugin has been used before to apply settings
        setIsApplying(false); // End UI loading state
        updateSettingsChanges(); // Evaluate if local settings differ from disk and show notification if changes detected
    };

    // define code editor
    const [editorCode, setEditorCode] = React.useState("");
    const [isEditorFullscreen, setIsEditorFullscreen] = React.useState(false);

    const codeEditor = (
        <div style={{ position: "relative" }}>
            <FullscreenToggle
                onToggle={() => setIsEditorFullscreen(!isEditorFullscreen)}
                isFullscreen={isEditorFullscreen}
                style={{
                    position: "absolute",
                    top: "8px",
                    right: "8px",
                    zIndex: 10
                }}
            />
            <CodeEditor
                code={editorCode}
                setCode={setEditorCode}
                updateSettingsChanges={updateSettingsChanges}
                exitFullScreen={() => setIsEditorFullscreen(false)}
                isFullscreen={isEditorFullscreen}
            />
            {isEditorFullscreen && (
                <Button
                    isDisabled={isApplying || !canApply}
                    id='apply_button'
                    variant='primary'
                    onClick={handleApplySettings}
                    style={{
                        position: "absolute",
                        bottom: "23px",
                        right: "30px",
                        zIndex: 10
                    }}
                >
                    {_("Apply Settings")}
                </Button>
            )}
        </div>
    );

    /**
     * Loads content into the code editor based on current settings hashes.
     * If hashes don't match, loads settings from file, otherwise loads from localDb.
     */
    const loadEditorContent = async () => {
        const simpleHash = getLocalDbValue("simpleHash");
        const advancedHash = getLocalDbValue("advancedHash");
        const settingsJsHash = await getSettingsJsChecksum();

        let fileContent = "";
        if (advancedHash !== settingsJsHash && settingsJsHash !== undefined && simpleHash !== settingsJsHash) {
            fileContent = await readSettingsJsContent();
        } else {
            fileContent = getLocalDbValue("advancedSettings");
        }
        setEditorCode(fileContent);
    };

    /**
     * Sets initial editor mode based on settings hash comparison.
     * Uses simple mode for default/no settings, advanced mode for custom settings.
     */
    const setInitialAdvancedMode = async () => {
        const simpleHash = getLocalDbValue("simpleHash");
        const advancedHash = getLocalDbValue("advancedHash");
        const settingsJsHash = await getSettingsJsChecksum();
        let initAdvancedMode = null;
        // code
        if (settingsJsHash === defaultSettingsJsHash && !advancedHash) {
            // simple settings is equal to nodereds default settings.js
            initAdvancedMode = false;
        } else if (settingsJsHash === undefined) {
            // no settings.js file present
            initAdvancedMode = false;
        } else if (simpleHash === settingsJsHash) {
            // settings.js has been editor via simple mode
            initAdvancedMode = false;
        } else {
            // settings.js has been edited via advanced mode or externally
            initAdvancedMode = true;
        }
        loadEditorContent();
        setUseAdvancedMode(initAdvancedMode);
    };

    React.useEffect(() => { // initial page setup
        if (!isLocalDbReady) return;
        updateSettingsChanges();
        setInitialAdvancedMode();
        const rawUsers = getLocalDbValue("users") || [];
        const sanitizedUsers = rawUsers.map(({ username, permissions }) => ({
            username,
            permissions
        }));
        setUsers(sanitizedUsers);
        setUseAuthentication(getLocalDbValue("useAuthentication"));
    }, [isLocalDbReady]);

    React.useEffect(() => {
        const fetchData = async () => {
            await init();
            setIsConfigured(anyHashExists());
            setIsLocalDbReady(true);
        };
        fetchData();
    }, []);

    React.useEffect(() => {
        const handleEscKey = (e) => {
            if (e.key === "Escape" && isEditorFullscreen) {
                setIsEditorFullscreen(false);
            }
        };

        window.addEventListener("keydown", handleEscKey);
        return () => {
            window.removeEventListener("keydown", handleEscKey);
        };
    }, [isEditorFullscreen]);

    React.useEffect(() => {
        const handleCanUseSerialInterfaces = async () => {
            const canUse = await canNoderedUseSerialInterfaces();
            setCanUseSerialInterfaces(canUse);
        };
        handleCanUseSerialInterfaces();
    }, []);

    React.useEffect(() => {
        const handleCanUseAudioVideo = async () => {
            const canUse = await canNoderedUseAudioVideo();
            setCanUseAudioVideo(canUse);
        };
        handleCanUseAudioVideo();
    }, []);

    React.useEffect(() => {
        const initialSandbox = async () => {
            const isProtected = await isSandbox();
            setProtectSystem(isProtected);
        };
        initialSandbox();
    }, []);

    if (!isAdmin) {
        // Show prompt to grant admin rights via polkit dialog if user lacks permissions
        return <AdminAccessButton />;
    }

    return (
        <NotificationProvider>
            <Notifications />
            <Page>
                {isEditorFullscreen
                    ? codeEditor
                    : (
                        <PageSection variant={PageSectionVariants.default}>
                            <Grid hasGutter>
                                <GridItem>
                                    <ServiceCard
                                        isConfigured={isConfigured}
                                        nodeRedState={nodeRedState}
                                        canUseSerialInterfaces={canUseSerialInterfaces}
                                        setCanUseSerialInterfaces={setCanUseSerialInterfaces}
                                        canUseAudioVideo={canUseAudioVideo}
                                        setCanUseAudioVideo={setCanUseAudioVideo}
                                        protectSystem={protectSystem}
                                        setProtectSystem={setProtectSystem}
                                        protectSystemLoading={protectSystemLoading}
                                        setProtectSystemLoading={setProtectSystemLoading}
                                    />
                                </GridItem>
                                <GridItem>
                                    <SettingsCard
                                        codeEditor={codeEditor}
                                        setEditorCode={setEditorCode}
                                        isLocalDbReady={isLocalDbReady}
                                        updateSettingsChanges={updateSettingsChanges}
                                        settingsChanged={settingsChanged}
                                        users={users}
                                        setUsers={setUsers}
                                        useAuthentication={useAuthentication}
                                        setUseAuthentication={setUseAuthentication}
                                        useAdvancedMode={useAdvancedMode}
                                        setUseAdvancedMode={setUseAdvancedMode}
                                        isApplying={isApplying}
                                        canApply={canApply}
                                        setCanApply={setCanApply}
                                        simpleSettingsValid={simpleSettingsValid}
                                        handleApplySettings={handleApplySettings}
                                    />
                                </GridItem>
                            </Grid>
                        </PageSection>
                    )}
            </Page>
        </NotificationProvider>
    );
};
