// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import {
    Button,
    Card,
    CardBody,
    CardHeader,
    CardTitle,
    Flex,
    FlexItem,
    Grid,
    GridItem,
    Switch,
    Text,
    TextContent,
    TextVariants
} from "@patternfly/react-core";
import React from "react";
import cockpit from "cockpit";
import { getHostName, getPortNumber, parseFilePath } from "../common/helper.js";
import { customFeatureRenderers } from "./config/featureRenderers.jsx";
import { useCustomization } from "../branding.js";
import { useAppConfig, useNotifications } from "../common/hooks.jsx";

const _ = cockpit.gettext;

/**
 * Display types for feature buttons in the UI.
 * Note: Translations are applied to dynamic string values, which requires
 * having translations in poupdate. Direct translation calls in this object
 * are not functional.
 */
export const DISPLAY_TYPES = {
    OPEN: _("Open"),
    CONFIGURE: _("Configure"),
    INSTALL: _("Install")
};
export const DISPLAY_TYPE_COLORS = {
    [DISPLAY_TYPES.OPEN]: "primary",
    [DISPLAY_TYPES.CONFIGURE]: "tertiary",
    [DISPLAY_TYPES.INSTALL]: "secondary"
};

export function AppCard () {
    const { addNotification } = useNotifications();
    const { productName } = useCustomization();

    const apps = [
        useAppConfig({
            name: "pictory",
            displayName: "PiCtory",
            packageNames: ["pictory", "pictory-apache"],
            displayType: DISPLAY_TYPES.OPEN,
            click: openPictoryHandler(addNotification),
            helpText: (
                <>
                    <p>
                        {cockpit.format(_("In PiCtory, you configure the hardware settings for modules connected to the $0. Here, you can define, arrange, and configure expansion modules like digital or analog I/O modules. This ensures the $0 correctly detects and communicates with them."), productName)}
                    </p>
                    <p>
                        <strong>{_("Tip")}:</strong>{" "}
                        {_("Remember to save your changes and reset the driver to apply the settings.")}
                    </p>
                </>
            )
        }),
        useAppConfig({
            name: "nodered",
            displayName: "Node-RED",
            packageNames: ["revpi-nodered", "revpi-nodered-proxy-apache", "noderedrevpinodes-server", "cockpit-revpi-nodered"],
            displayType: DISPLAY_TYPES.CONFIGURE,
            click: () => { cockpit.jump("/revpi-nodered") },
            helpText: (
                <p>
                    {_("A simple flow-based tool for connecting devices and automating workflows.")}
                </p>
            )
        })
    ];

    const handleAppAction = (app) => {
        if (!app.installer.isInstalled) {
            app.installer.handleInstall?.();
        } else {
            app.click?.();
        }
    };

    return (
        <Card isFullHeight name='app-list'>
            <CardTitle style={{ fontWeight: "bold" }}>
                {cockpit.format("$0 Apps", productName)}
            </CardTitle>
            <CardBody>
                <Grid hasGutter>
                    {apps.map((app) => (
                        <GridItem key={app.name} span={12}>
                            <div
                                style={{
                                    padding: "1rem",
                                    borderRadius: "var(--pf-global--BorderRadius--lg)",
                                    border: "1px solid #e5e5e5",
                                    // backgroundColor: "#fcfcfc",
                                    boxShadow: "0 1px 2px rgba(0, 0, 0, 0.03)",
                                    transition: "all 0.2s ease"
                                }}
                                onMouseEnter={(e) => {
                                    e.currentTarget.style.backgroundColor = "#f7f7f7";
                                    e.currentTarget.style.boxShadow = "0 2px 4px rgba(0, 0, 0, 0.06)";
                                }}
                                onMouseLeave={(e) => {
                                    e.currentTarget.style.backgroundColor = "#ffffff";
                                    e.currentTarget.style.boxShadow = "0 1px 2px rgba(0, 0, 0, 0.03)";
                                }}
                            >
                                <Flex alignItems={{ default: "alignItemsFlexStart" }} justifyContent={{ default: "justifyContentSpaceBetween" }} gap={{ default: "gapMd" }}>
                                    <FlexItem grow={{ default: "grow" }}>
                                        <TextContent>
                                            <Text component={TextVariants.h3} style={{ fontWeight: 500, marginBottom: "0.25rem" }}>
                                                {app.displayName}
                                            </Text>
                                            <div style={{ fontSize: "0.85rem", color: "#555", lineHeight: 1.4 }}>
                                                {app.helpText}
                                            </div>
                                        </TextContent>
                                    </FlexItem>
                                    <FlexItem>
                                        <Button
                                            variant={
                                                app.installer.isInstalled
                                                    ? DISPLAY_TYPE_COLORS[app.displayType]
                                                    : DISPLAY_TYPE_COLORS.install
                                            }
                                            onClick={() => handleAppAction(app)}
                                            style={{ minWidth: "115px" }}
                                            isDisabled={app.installer.isInstalling}
                                        >
                                            <span style={{ fontWeight: "bold" }}>
                                                {app.installer.isInstalled ? _(app.displayType) : _(DISPLAY_TYPES.INSTALL)}
                                            </span>
                                        </Button>
                                    </FlexItem>
                                </Flex>
                            </div>
                        </GridItem>
                    ))}
                </Grid>
            </CardBody>
        </Card>
    );
}

const openPictoryHandler = (addNotification) => {
    return async () => {
        try {
            const token = await createSsoToken();
            const url = `https://${getHostName()}:${getPortNumber()}/pictory/php/sso_login.php?sso_token=${token}`;
            window.open(url, "_blank");
        } catch (error) {
            addNotification(_("Could not launch PiCtory due to a login token error."), "", "danger");
            console.error("Error while creating SSO token", error);
        }
    };
};

// Function to create an SSO token by checking multiple script locations
const createSsoToken = async () => {
    const filePath = await parseFilePath("scripts/create_sso_token.sh");
    try {
        return await cockpit.spawn(filePath, { superuser: "require" });
    } catch (error) {
        console.log(`No script found at '${filePath}'.`, error);
    }
};

/**
 * Renders a card component that groups and displays a list of features. Each feature
 * includes various attributes such as title, state, availability, and an optional
 * custom renderer. The card also provides interactive controls (e.g., switches) to
 * toggle the state of individual features.
 *
 * @param {Object} props - The properties object.
 * @param {string} props.title - The title of the feature group to display.
 * @param {Array} props.features - The array of feature objects to be displayed in the card.
 * Each feature must include properties such as `id`, `type`, `title`, and optionally `helpText` or `options`.
 * @param {Object} props.featureStates - An object that maps feature IDs to their enabled/disabled state.
 * @param {Object} props.availableFeatures - An object that maps feature IDs to their availability status.
 * @param {Function} props.toggle - A callback function triggered when a feature toggle switch is changed.
 * Receives two arguments: the feature ID and the new state (enabled/disabled).
 *
 * @return {JSX.Element} Returns the rendered feature group card component.
 */
export function FeatureGroupCard ({ title, features, featureStates, availableFeatures, toggle, gridColumn = "span 1", gridRow }) {
    const typeColors = {
        config: "#06c",
        service: "#3e8635",
        default: "#8a8d90"
    };

    return (
        <div style={{ gridColumn, gridRow }}>
            <Card
                isCompact
                style={{
                    height: "100%", // ⬅️ Ensures the card fills the grid row height
                    boxShadow: "var(--pf-global--BoxShadow--sm)",
                    display: "flex",
                    flexDirection: "column"
                }}
            >
                <CardHeader>
                    <h2 style={{ fontSize: "1rem", fontWeight: 600, margin: 0 }}>{title}</h2>
                </CardHeader>
                <CardBody>
                    <div
                        style={{
                            display: "grid",
                            gridTemplateColumns: "repeat(auto-fit, minmax(280px, 1fr))",
                            gap: "1rem"
                        }}
                    >
                        {features.map((feature) => {
                            const isEnabled = featureStates[feature.id] ?? false;
                            const isAvailable = availableFeatures[feature.id] ?? false;
                            const color = typeColors[feature.type] || typeColors.default;
                            const customRenderer = customFeatureRenderers[feature.id];

                            if (customRenderer) {
                                return (
                                    <div key={feature.id}>
                                        {customRenderer(feature, isEnabled, toggle, isAvailable)}
                                    </div>
                                );
                            }

                            return (
                                <div
                                    key={feature.id}
                                    style={{
                                        padding: "1rem",
                                        borderLeft: `4px solid ${color}`,
                                        border: "1px solid #f5f5f5",
                                        borderRadius: "5px",
                                        backgroundColor: isAvailable ? "#fff" : "#f5f5f5",
                                        opacity: isAvailable ? 1 : 0.6,
                                        display: "flex",
                                        flexDirection: "column",
                                        justifyContent: "space-between",
                                        minHeight: "160px"
                                    }}
                                >
                                    <div style={{ marginBottom: "0.5rem" }}>
                                        <div style={{ fontWeight: 500, fontSize: "1rem", color }}>
                                            {feature.title}
                                        </div>

                                        {feature.helpText && (
                                            <div style={{ fontSize: "0.85rem", color: "#444", marginBottom: "0.5rem" }}>
                                                {feature.helpText}
                                            </div>
                                        )}

                                        {feature.options && (
                                            <div style={{ fontSize: "0.85rem", fontStyle: "italic", color: "#666" }}>
                                                {isEnabled
                                                    ? feature.options.enabled?.title
                                                    : feature.options.disabled?.title}
                                            </div>
                                        )}
                                    </div>

                                    <Switch
                                        id={`switch-${feature.id}`}
                                        label={isAvailable ? (isEnabled ? "On" : "Off") : "Not supported"}
                                        isChecked={isEnabled}
                                        isDisabled={!isAvailable}
                                        onChange={(_event, checked) => {
                                            toggle(feature.id, checked);
                                        }}
                                    />
                                </div>
                            );
                        })}
                    </div>
                </CardBody>
            </Card>
        </div>
    );
}
