// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";
import React from "react";
import { ClipboardCopy } from "@patternfly/react-core";
import { useLoggedInUser } from "../../../pkg/lib/hooks.js";
import { getSshAddress } from "../../common/helper.js";
import { useCustomization } from "../../branding.js";

const _ = cockpit.gettext;

export function useConfigData () {
    const user = useLoggedInUser();
    const { productName } = useCustomization();
    const sshAddress = React.useMemo(() => {
        if (!user) return "";
        return getSshAddress(user);
    }, [user]);
    return React.useMemo(() => [
        {
            id: "systemd-timesyncd",
            title: _("Set clock from NTP-Server"),
            type: "config",
            helpText: (
                <div className='help-text'>
                    <p>{_("This setting synchronizes your system time with an NTP server.")}</p>
                </div>
            )
        },
        {
            id: "ssh",
            title: _("SSH daemon to enable remote login"),
            type: "config",
            helpText: (
                <div className='help-text'>
                    <p>{cockpit.format(_("SSH lets you securely access this $0’s terminal over the network."), productName)}</p>
                    <p>{_("Connect via SSH with")}: </p>
                    <ClipboardCopy
                        isReadOnly
                        style={{ marginTop: "0.5rem" }}
                    >
                        ssh {sshAddress}
                    </ClipboardCopy>
                </div>
            )
        },
        {
            id: "gui",
            title: _("Boot to GUI"),
            type: "config",
            rebootRequired: true,
            helpText: (
                <div className='help-text'>
                    <p>{cockpit.format(_("In desktop mode, you can boot the $0 directly into a graphical user interface as an alternative to the terminal."), productName)}</p>
                </div>
            )
        },
        {
            id: "var-log.mount",
            title: _("Store logs on"),
            type: "config",
            rebootRequired: true,
            options: {
                enabled: {
                    title: _("Temporary storage")
                },
                disabled: {
                    title: _("Flash storage")
                }
            },
            helpText: (
                <div className='help-text'>
                    <p>{_("Configures where system logs are stored.")}</p>
                    <div>
                        <strong>{_("Storage Options")}:</strong>
                        <div>
                            <p>
                                <strong>{_("Temporary storage")}:</strong>{" "}
                                {_(
                                    "Logs are stored in RAM and lost on reboot. Reduces flash wear but makes debugging harder."
                                )}
                            </p>
                            <p>
                                <strong>{_("Flash storage")}:</strong>{" "}
                                {_(
                                    "Logs are stored persistently. Useful for debugging."
                                )}
                            </p>
                        </div>
                    </div>
                </div>
            )
        },
        {
            id: "bluetooth",
            title: _("BT"),
            type: "config",
            rebootRequired: false,
            helpText: (
                <div className='help-text'>
                    <p>{_("Enable BT for using Bluetooth peripherals such as keyboards or audio devices.")}</p>
                </div>
            )
        },
        {
            id: "ieee80211",
            title: _("WLAN"),
            type: "config",
            rebootRequired: false
        },
        {
            id: "external-antenna",
            title: _("Select antenna"),
            type: "config",
            rebootRequired: true,
            options: {
                enabled: {
                    title: _("External antenna (SMA)")
                },
                disabled: {
                    title: _("Internal antenna")
                }
            }
        },
        {
            id: "revpipyload",
            title: "RevPiPyLoad",
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{cockpit.format(_("A Python-based runtime environment for $0."), productName)}</p>
                </div>
            )
        },
        {
            id: "avahi",
            title: _("Avahi (ZeroConf)"),
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{_("Avahi enables ZeroConf networking, allowing easy discovery of network services.")}</p>
                </div>
            ),
            hoverText: {
                isDisabled: _("You cannot disable Avahi while connected to Cockpit via an Avahi address (.local). If you need to disable it, please use the command line or connect using the device's IP address instead.")
            }
        },
        {
            id: "dphys-swapfile",
            title: _("Swap / page file (recommended for GUI)"),
            type: "service",
            helpText: (
                <div className='help-text'>
                    <p>{_("Swap uses virtual memory when RAM is full, helping prevent system crashes.")}</p>
                </div>
            )
        }
    ], [sshAddress]);
}
