// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import cockpit from "cockpit";
import { HelpText, UiCardTitle, UiDivider } from "../../common/ui-components.jsx";
import {
    listSupportedLocales,
    listTimezones,
    currentLocale,
    setLocale,
    currentTimezone,
    setTimezone,
    getCurrentKeyboardLayout,
    setKeyboardLayout, getTimezoneBuckets, bucketForTimezone, getAllKeyboardLayouts
} from "../../common/helper.js";
import {
    Alert,
    Card,
    CardBody,
    CardTitle,
    Form,
    FormGroup,
    FormSelect,
    FormSelectOption, Grid,
    GridItem,
    Spinner
} from "@patternfly/react-core";
import { SearchSelect } from "../revpi-config.components.jsx";
import { ZoneIcon } from "@patternfly/react-icons";

const _ = cockpit.gettext;

export function RegionalSettingsCard () {
    const cardTitle = _("Regional Settings");

    // unified width for all selects (incl. SearchSelect wrapper)
    const SELECT_WIDTH = 300;

    // current values
    const [langCur, setLangCur] = React.useState("");
    const [tzCur, setTzCur] = React.useState("");
    const [kbCur, setKbCur] = React.useState("");

    // selections
    const [langSel, setLangSel] = React.useState("");
    const [tzBucketSel, setTzBucketSel] = React.useState("");
    const [tzZoneSel, setTzZoneSel] = React.useState("");
    const [kbSel, setKbSel] = React.useState("");

    // lists
    const [locales, setLocales] = React.useState([]);
    const [tzBuckets, setTzBuckets] = React.useState([]);
    const [tzZonesForBucket, setTzZonesForBucket] = React.useState([]);
    const [kbLayouts, setKbLayouts] = React.useState([]);

    // UI state
    const [loading, setLoading] = React.useState(true);
    const [busy, setBusy] = React.useState({ lang: false, tz: false, kb: false });
    const [message, setMessage] = React.useState(null);

    React.useEffect(() => {
        let alive = true;
        (async () => {
            try {
                const [lcList, tzList, lcCur, tzNow, kbNow, kbAll] = await Promise.all([
                    listSupportedLocales(),
                    listTimezones(),
                    currentLocale(),
                    currentTimezone(),
                    getCurrentKeyboardLayout(),
                    getAllKeyboardLayouts()
                ]);
                if (!alive) return;

                // locales
                setLocales(lcList);
                setLangCur(lcCur || "");
                setLangSel(lcCur || "");

                // keyboard
                setKbLayouts(kbAll || []);
                setKbCur(kbNow || "");
                setKbSel(kbNow || "");

                // timezones via buckets
                setTzCur(tzNow || "");
                const buckets = getTimezoneBuckets(tzList);
                setTzBuckets(buckets);

                const curBucket = bucketForTimezone(tzNow || "");
                const initialBucket = curBucket || (buckets[0]?.name || "");
                setTzBucketSel(initialBucket);

                const zonesForBucket = (buckets.find(b => b.name === initialBucket) || {}).zones || [];
                setTzZonesForBucket(zonesForBucket);

                const initialZone = zonesForBucket.includes(tzNow) ? tzNow : (zonesForBucket[0] || "");
                setTzZoneSel(initialZone);
            } catch (e) {
                if (!alive) return;
                setMessage({ variant: "danger", text: _("Could not read regional settings: ") + (e?.message || e) });
            } finally {
                if (alive) setLoading(false);
            }
        })();
        return () => { alive = false };
    }, []);

    // Locale
    const handleLocaleSelect = React.useCallback(async (value) => {
        setLangSel(value);
        if (!value) return;
        setBusy(b => ({ ...b, lang: true }));
        setMessage(null);
        try {
            await setLocale(value);
            setLangCur(value);
            setMessage({ variant: "success", text: cockpit.format(_("System language set to $0"), value) });
        } catch (e) {
            setMessage({ variant: "danger", text: cockpit.format(_("Failed to set system language: $0"), (e?.message || e)) });
        } finally {
            setBusy(b => ({ ...b, lang: false }));
        }
    }, []);

    const toLocaleLabel = (code) => {
        try {
            const [lang, region] = code.split(/[-_]/);
            const dnLang = new Intl.DisplayNames([code, "en"], { type: "language" });
            const dnReg = new Intl.DisplayNames([code, "en"], { type: "region" });
            const langName = dnLang.of(lang);
            const regName = region ? dnReg.of(region.toUpperCase()) : undefined;
            return regName ? `${code} ${langName} (${regName})` : `${code} ${langName}`;
        } catch {
            return code;
        }
    };
    const localeOptions = React.useMemo(() => locales.map(l => toLocaleLabel(l)), [locales]);

    // Timezone (Buckets)
    const onBucketChange = React.useCallback((_evt, newBucket) => {
        setTzBucketSel(newBucket);
        const zones = (tzBuckets.find(b => b.name === newBucket) || {}).zones || [];
        setTzZonesForBucket(zones);
        const keep = zones.includes(tzZoneSel) ? tzZoneSel : (zones[0] || "");
        setTzZoneSel(keep);
    }, [tzBuckets, tzZoneSel]);

    const onZoneChange = React.useCallback(async (_evt, fullTz) => {
        setTzZoneSel(fullTz);
        if (!fullTz) return;
        setBusy(b => ({ ...b, tz: true }));
        setMessage(null);
        try {
            await setTimezone(fullTz);
            setTzCur(fullTz);
            setMessage({ variant: "success", text: cockpit.format(_("Time zone set to $0"), fullTz) });
        } catch (e) {
            setMessage({ variant: "danger", text: cockpit.format(_("Failed to set time zone: $0"), (e?.message || e)) });
        } finally {
            setBusy(b => ({ ...b, tz: false }));
        }
    }, []);

    // Keyboard
    const onKeyboardChange = React.useCallback(async (_evt, layout) => {
        setKbSel(layout);
        if (!layout) return;
        setBusy(b => ({ ...b, kb: true }));
        setMessage(null);
        try {
            await setKeyboardLayout(layout);
            const kbNow = await getCurrentKeyboardLayout();
            setKbCur(kbNow);
            setMessage({ variant: "success", text: cockpit.format(_("Keyboard layout set to $0"), layout) });
        } catch (e) {
            setMessage({ variant: "danger", text: cockpit.format(_("Failed to set keyboard layout: $0"), (e?.message || e)) });
        } finally {
            setBusy(b => ({ ...b, kb: false }));
        }
    }, []);

    // Format zone labels without the category (after first "/"), prettify underscores
    const tzLabel = (fullId) => {
        const parts = String(fullId).split("/");
        return parts.slice(1).join("/").replace(/_/g, " ") || fullId; // "Europe/Berlin" -> "Berlin"
    };

    if (loading) {
        return (
            <Card isFullHeight name='regional-settings'>
                <CardTitle>
                    <UiCardTitle>
                        <ZoneIcon style={{ marginRight: "1rem" }} />
                        {cardTitle}
                    </UiCardTitle>
                    <UiDivider />
                </CardTitle>
                <CardBody style={{ display: "flex", justifyContent: "center", alignItems: "center" }}>
                    <Spinner size='xl' />
                </CardBody>
            </Card>
        );
    }

    return (
        <Card isFullHeight name='regional-settings'>
            <CardTitle>
                <UiCardTitle>
                    <ZoneIcon style={{ marginRight: "1rem" }} />
                    {cardTitle}
                </UiCardTitle>
                <UiDivider />
            </CardTitle>
            <CardBody>
                {message && (
                    <div style={{ marginBottom: 12 }}>
                        <Alert variant={message.variant} title={message.text} isInline />
                    </div>
                )}

                <Grid hasGutter>
                    {/* Time Zone */}
                    <GridItem span={12}>
                        <Form isWidthLimited>
                            <FormGroup label={_("Time Zone")} fieldId='timezone'>
                                <div style={{ display: "flex", alignItems: "center", gap: 8, flexWrap: "wrap", justifyContent: "space-between", width: "100%" }}>
                                    {/* Left: bucket + zone with single spinner */}
                                    <div style={{ display: "flex", alignItems: "center", gap: 8, flexWrap: "wrap", width: SELECT_WIDTH }}>
                                        {/* Bucket */}
                                        <FormSelect
                                            id='tz-bucket'
                                            value={tzBucketSel}
                                            onChange={onBucketChange}
                                            isDisabled={busy.tz}
                                            style={{ width: SELECT_WIDTH }}
                                        >
                                            {tzBuckets.map(b => (
                                                <FormSelectOption key={b.name} value={b.name} label={b.name} />
                                            ))}
                                        </FormSelect>

                                        {/* Zone (full values, short labels) */}
                                        <FormSelect
                                            id='tz-zone'
                                            value={tzZoneSel}
                                            onChange={onZoneChange}
                                            isDisabled={busy.tz}
                                            style={{ width: SELECT_WIDTH }}
                                        >
                                            {tzZonesForBucket.map(z => (
                                                <FormSelectOption key={z} value={z} label={tzLabel(z)} />
                                            ))}
                                        </FormSelect>
                                    </div>
                                    {busy.tz && <Spinner size='lg' />}

                                    {/* Right: current */}
                                    <span style={{ opacity: 0.8, marginLeft: "auto" }}>
                                        {_("Current: ")}<code>{tzCur || "-"}</code>
                                    </span>
                                </div>
                            </FormGroup>
                        </Form>
                    </GridItem>

                    {/* Language (Locale) */}
                    <GridItem span={12}>
                        <Form isWidthLimited>
                            <FormGroup label={_("System language")} fieldId='locale'>
                                <div style={{ display: "flex", alignItems: "center", gap: 8, flexWrap: "wrap", justifyContent: "space-between", width: "100%" }}>
                                    <div style={{ display: "flex", alignItems: "center", gap: 6 }}>
                                        <div style={{ width: SELECT_WIDTH, flex: "0 0 auto" }}>
                                            <SearchSelect
                                                options={localeOptions}
                                                status={(langSel || langCur || "").trim()}
                                                handleSelect={handleLocaleSelect}
                                                isDisabled={busy.lang}
                                                placeholder={_("Select system language...")}
                                            />
                                        </div>
                                        {busy.lang && <Spinner size='lg' />}
                                    </div>

                                    <span style={{ opacity: 0.8, marginLeft: "auto" }}>
                                        {_("Current: ")}<code>{langCur || "-"}</code>
                                    </span>
                                </div>
                            </FormGroup>
                        </Form>
                    </GridItem>

                    {/* Keyboard */}
                    <GridItem span={12}>
                        <Form isWidthLimited>
                            <FormGroup label={_("Keyboard Layout")} fieldId='keyboard'>
                                <div style={{ display: "flex", alignItems: "center", gap: 8, flexWrap: "wrap", justifyContent: "space-between", width: "100%" }}>
                                    <div style={{ display: "flex", alignItems: "center", gap: 6 }}>
                                        <FormSelect
                                            id='keyboard'
                                            value={kbSel}
                                            onChange={onKeyboardChange}
                                            isDisabled={busy.kb}
                                            style={{ width: SELECT_WIDTH }}
                                        >
                                            <FormSelectOption value='' label={_("Select...")} />
                                            {kbLayouts.map(k => (
                                                <FormSelectOption key={k} value={k} label={k} />
                                            ))}
                                        </FormSelect>
                                        {busy.kb && <Spinner size='lg' />}
                                    </div>
                                    <HelpText>{_("Language setting for a USB keyboard that is connected directly to the RevPi.")}</HelpText>
                                    <span style={{ opacity: 0.8, marginLeft: "auto" }}>
                                        {_("Current: ")}<code>{kbCur || "-"}</code>
                                    </span>
                                </div>
                            </FormGroup>
                        </Form>
                    </GridItem>
                </Grid>
            </CardBody>
        </Card>
    );
}
