// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import { useEffect, useState } from "react";
import { readFromFile } from "./common/helper.js";

let cachedCustomization = null;

/**
 * A custom React hook for retrieving customization data from a configuration file.
 * This hook reads a predefined JSON file to fetch customization data, with fallback logic
 * to default values if the file is not found or cannot be read.
 *
 * It uses a stateful approach by caching the customization and updating only when necessary.
 * The hook ensures the customization data is available for use in React components.
 *
 * @function
 * @returns {Object} An object containing the customization data.
 * @property {Object} customization - The customization data fetched from the configuration file or default values.
 */
export const useCustomization = () => {
    const customizationFilePath = "/usr/share/cockpit/branding/debian/customization.json";
    const [customization, setCustomization] = useState(cachedCustomization);

    // Read and parse the customization file, returns an empty object if file is not found or invalid
    const readCustomization = async () => {
        try {
            const data = await readFromFile(customizationFilePath);
            return JSON.parse(data);
        } catch (error) {
            console.error("Error loading customization file:", error);
            return {};
        }
    };

    // Load customization data or use defaults, caches result for subsequent calls
    const loadCustomizationData = async () => {
        const defaults = {
            productName: "RevPi"
        };

        if (cachedCustomization !== null) {
            return cachedCustomization;
        }

        try {
            const config = await readCustomization();
            const customizationData = { ...defaults, ...config };
            cachedCustomization = customizationData;
            setCustomization(customizationData);
        } catch {
            cachedCustomization = defaults;
            setCustomization(defaults);
        }
    };

    // Load customization on initial render if not already loaded
    useEffect(() => {
        if (customization === null) {
            loadCustomizationData();
        }
    }, []);

    return { ...customization };
};
